/**
 * Copyright (c) 2007-2009, Opera Software ASA
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of Opera Software ASA nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY OPERA SOFTWARE ASA AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL OPERA SOFTWARE ASA AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

// Twitter API: http://apiwiki.twitter.com/REST+API+Documentation
// Not all methods are implemented yet

// Note: this file needs request.js to work

function TwitterApi(settings) {
  // Settings
  var source = settings.source || '';
  var format = '.' + (settings.format || 'json');
  
  arguments.callee.MAX_CHARS = 140; // static constant
  
  return {
    login: function(username, password, callback, errorCallback) {
      // Save the credentials to use in later requests with Basic Authentication
      this.username = username;
      this.password = password;
      
      this.verifyCredentials(username, password, callback, errorCallback);
    },
    
    ////////////////////////////////////////////////////////////////////////////
    // API methods
    ////////////////////////////////////////////////////////////////////////////
    
    // Status Methods
    
    publicTimeline: function(callback, errorCallback) {
      makeRequest({
        method: 'GET',
        url: 'http://twitter.com/statuses/public_timeline' + format,
        callback: callback,
        errorCallback: errorCallback
      }, true);
    },
    
    
    friendsTimeline: function(callback, errorCallback, options) {
      var query = formatQuery(options);
      
      makeRequest({
        method: 'GET',
        url: 'https://twitter.com/statuses/friends_timeline' + format + query,
        headers: {
          'Cache-Control': 'no-cache'
        },
        username: this.username,
        password: this.password,
        callback: callback,
        errorCallback: errorCallback
      }, true);
    },
    
    
    userTimeline: function(callback, errorCallback, options) {
      var query = formatQuery(options);
      
      makeRequest({
        method: 'GET',
        url: 'https://twitter.com/statuses/user_timeline' + format + query,
        headers: {
          'Cache-Control': 'no-cache'
        },
        username: this.username,
        password: this.password,
        callback: callback,
        errorCallback: errorCallback
      }, true);
    },
    
    
    show: function(id, callback, errorCallback) {
      makeRequest({
        method: 'GET',
        url: 'https://twitter.com/statuses/show/' + id + format,
        username: this.username,
        password: this.password,
        callback: callback,
        errorCallback: errorCallback
      }, true);
    },
    
    
    update: function(status, inReplyTo, callback, errorCallback) {
      var query = formatQuery({ 'in_reply_to_status_id': inReplyTo, 'source': source });
      
      makeRequest({
        method: 'POST',
        url: 'https://twitter.com/statuses/update' + format + query,
        headers: {
          'Content-Type': 'application/x-www-form-urlencoded'
        },
        data: 'status=' + encodeURIComponent(status),
        username: this.username,
        password: this.password,
        callback: callback,
        errorCallback: errorCallback
      });
    },
    
    
    // Named 'replies' in the official API. 'repliesTimeline' here for consistency.
    repliesTimeline: function(callback, errorCallback) {
      makeRequest({
        method: 'GET',
        url: 'https://twitter.com/statuses/replies' + format,
        username: this.username,
        password: this.password,
        callback: callback,
        errorCallback: errorCallback
      }, true);
    },
    
    
    destroy: function(id, callback, errorCallback) {
      makeRequest({
        method: 'DELETE',
        url: 'https://twitter.com/statuses/destroy/' + id + format,
        username: this.username,
        password: this.password,
        callback: callback,
        errorCallback: errorCallback
      });
    },
    
    
    // User methods
    
    friends: function(callback, errorCallback) {
      makeRequest({
        method: 'GET',
        url: 'https://twitter.com/statuses/followers' + format,
        username: this.username,
        password: this.password,
        callback: callback,
        errorCallback: errorCallback
      });
    },
    
    
    // Direct message methods
    
    directTimeline: function(callback, errorCallback) {
      makeRequest({
        method: 'GET',
        url: 'https://twitter.com/direct_messages' + format,
        username: this.username,
        password: this.password,
        callback: callback,
        errorCallback: errorCallback
      });
    },
    
    
    // Account Methods
    
    verifyCredentials: function(username, password, callback, errorCallback) {
      makeRequest({
        method: 'HEAD',
        url: 'https://twitter.com/account/verify_credentials' + format,
        username: username,
        password: password,
        callback: callback,
        errorCallback: errorCallback
      });
    },
    
    endSession: function(callback) {
      makeRequest({
        method: 'POST',
        url: 'http://twitter.com/account/end_session',
        callback: callback,
        errorCallback: callback
      });
    },
    
    
    // Friendship Methods
    
    createFriendship: function(id, callback, errorCallback) {
      makeRequest({
        method: 'POST',
        url: 'https://twitter.com/friendships/create/' + id + format,
        username: this.username,
        password: this.password,
        callback: callback,
        errorCallback: errorCallback
      });
    },
    
    
    destroyFriendship: function(id, callback, errorCallback) {
      makeRequest({
        method: 'DELETE',
        url: 'https://twitter.com/friendships/destroy/' + id + format,
        username: this.username,
        password: this.password,
        callback: callback,
        errorCallback: errorCallback
      });
    },
    
    
    friendshipExists: function(id, callback, errorCallback) {
      makeRequest({
        method: 'POST',
        url: 'https://twitter.com/friendships/exists/' + id + format,
        username: this.username,
        password: this.password,
        callback: callback,
        errorCallback: errorCallback
      });
    },
    
    
    // Twitter search API
    
    search: function(query, callback, errorCallback, options) {
      if (query[0] != '?') query = formatQuery({ 'q': query });
      query += '&rpp=' + options.count;
      
      makeRequest({
        method: 'GET',
        url: 'http://search.twitter.com/search.json' + query,
        callback: callback,
        errorCallback: errorCallback
      }, true);
    },
    
    
    getTrends: function(callback, errorCallback) {
      makeRequest({
        method: 'GET',
        url: 'http://search.twitter.com/trends.json',
        callback: callback,
        errorCallback: errorCallback
      });
    } 
  };
  
  
  function formatQuery(parameters) {
    var query = [];
    for (var parameter in parameters) {
      var value = parameters[parameter];
      if (value !== null && value !== undefined) {
        query.push(encodeURIComponent(parameter) + '=' + encodeURIComponent(value));
      }
    }
    return '?' + query.join('&');
  }
}
