/* $Id: pngcodec.cpp 15435 2009-02-09 21:34:22Z peter1138 $ */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "minipng.hpp"

int Usage()
{
	printf("Usage:\n");
	printf(" to list properties:      pngcodec l <filename>\n");
	printf(" to clear all properties: pngcodec c <filename>\n");
	printf(" to replace properties:   pngcodec r <filename> [key=value] [key=value] [...]\n");
	printf(" to add properties:       pngcodec a <filename> [key=value] [key=value] [...]\n");
	return 1;
}

void PrintProperties(PNGFile& f)
{
	/* First dump the filename so that the output can easily be used to backup
	 * the png properties to a file such that they can easily be restored. */
	printf(f.filename);
	for (int q = 0; q < f.GetPropCount(); q++) {
		char *val = reinterpret_cast<char*>(malloc(f.GetPropValueLen(q) + 1));
		const char *key = f.GetProp(q, val);
		printf(" %s=%s", key, val);
		free(val);
	}
	/* end of line */
	printf("\n");
}

/**
 * Will read the PNG file and will print to stdout the tEXt properties in the PNG file.
 */
int List(const char *filename)
{
	PNGFile f;
	/* Read the png file */
	PNGRetVal rv = f.Read(filename);

	/* Process the return value */
	switch (rv) {
		default: fprintf(stderr, "Unknown error"); return rv;
		case PNG_FileNotFound: fprintf(stderr, "Cannot Read file"); return rv;
		case PNG_NotPNG: fprintf(stderr, "Invalid PNG file"); return rv;
		case PNG_OK: break;
	}

	/* Print the properties */
	PrintProperties(f);
	return 0;
}

/**
 * Will read the PNG file, modify the tEXt properties, save the file and will
 * print the new complete set to stdout.
 */
int Edit(const char *filename, int propc, const char **propv, bool Resetfirst)
{
	PNGFile f;
	/* Read the png file */
	PNGRetVal rv = f.Read(filename);

	/* Process the return value */
	switch (rv) {
		default: fprintf(stderr, "Unknown error"); return rv;
		case PNG_FileNotFound: fprintf(stderr, "Cannot Read file"); return rv;
		case PNG_NotPNG: fprintf(stderr, "Invalid PNG file"); return rv;
		case PNG_OK: break;
	}

	/* Remove all existing properties if this is requested */
	if (Resetfirst) f.ResetProps();

	/* Set all passes properties, if any */
	for (int i = 0; i < propc; i++) {
		/* Split the key=value in 2 zero terminated strings */
		char *prop = strdup(propv[i]);
		char *eq = strchr(prop, '=');
		/* Property must be in key=value or key= format. if not, cancel */
		if (!eq) { return Usage(); }

		*eq = '\0';
		char *val = eq + 1;
		f.SetProp(prop, val);
	}

	/* Write the PNG file with the changed properties */
	rv = f.Write();

	/* Process the return value */
	switch (rv)
	{
		default: fprintf(stderr, "Unknown Write error"); return rv;
		case PNG_AccessDenied: fprintf(stderr, "Cannot open file for writing"); return rv;
		case PNG_WriteError: fprintf(stderr, "Write error. Disk full?"); return rv;
		case PNG_OK: break;
	}

	/* Print the new properties */
	PrintProperties(f);

	return 0;
}

int main(int argc, const char **argv)
{
	/* A minimum of 2 arguments, command and filename, are required */
	if (argc < 3) return Usage();

	/* Get the first parameter, the command */
	char command = argv[1][0];

	/* The command must be 1 letter only */
	if (argv[1][1] != '\0') return Usage();

	/* The second argument is the filename */
	const char *filename = argv[2];

	switch (command) {
		case 'l': return argc == 3 ? List(filename) : Usage();
		case 'c': return argc == 3 ? Edit(filename, 0, 0, true) : Usage();
		case 'r': return argc > 3 ? Edit(filename, argc - 3, argv + 3, true) : Usage();
		case 'a': return argc > 3 ? Edit(filename, argc - 3, argv + 3, false) : Usage();
		default: return Usage();
	}
}
