/*
 * Generic LED driver for Technicolor Linux Gateway platforms.
 *
 * Copyright (C) 2013 Technicolor <linuxgw@technicolor.com>
 *
 */

#define LED_SHIFT_TASK_HZ 100

#include <linux/delay.h>
#include <linux/types.h>
#include <linux/device.h>
#include <linux/leds.h>
#include <linux/module.h>
#include <linux/spinlock.h>
#include <linux/gpio.h>
#include <linux/platform_device.h>
#include <linux/semaphore.h>
#include <linux/interrupt.h>
#include <linux/time.h>
#include <linux/workqueue.h>
#include <linux/jiffies.h>

#include "leds.h"
#include "board_led_defines.h"
#include "bcm_led.h"

static unsigned long led_bmp = 0;
static unsigned long last_led_bmp = 0;

static struct shiftled_led_platform_data *shift_pdata = NULL;

static struct workqueue_struct *shift_work_queue = NULL;
static struct delayed_work shift_work;

static struct platform_device gpled_device = {
	.name		       = "leds-gpio",
	.id		       = 0,
};


static void led_release(struct device *dev)
{
}

static struct platform_device shiftled_device = {
       .name                   = "shiftled-led",
       .id                     = 0,
       .dev = {
		.release = led_release,
	}
};

static void led_load_shift_reg(struct work_struct *pwork )
{
	int i;
	const unsigned long bitmap = led_bmp;
	struct shiftled_led_platform_data *pdata = shift_pdata;

	// Only need to shift the leds if there is a change
	if ( ( led_bmp != last_led_bmp ) && ( pdata != NULL  ) )
	{

		for (i = 0; i < pdata->reg_size; i++) { /* LSB first */
			gpio_set_value(pdata->reg_clk, 0);
			udelay(1);
			gpio_set_value(pdata->reg_data, (bitmap & (1<<i)) ? 1:0 );
			udelay(1);
			gpio_set_value(pdata->reg_clk, 1);
			udelay(2);
		}
		gpio_set_value(pdata->reg_rck, 1);
		udelay(2);
		gpio_set_value(pdata->reg_rck, 0);

		last_led_bmp = led_bmp;

	}

	if ( queue_delayed_work(shift_work_queue ,&shift_work, msecs_to_jiffies( 1000 / LED_SHIFT_TASK_HZ )) == 0 )
	{
		printk( KERN_INFO "LED Error adding work to workqueue" );
	}

	
}

static void shiftled_led_set(struct led_classdev *led_cdev,
                             enum led_brightness value)
{
	struct shiftled_led_data *led_dat =
		container_of(led_cdev, struct shiftled_led_data, cdev);
	unsigned long led_bmp_prev = led_bmp;

	led_dat->brightness = value;

	if (led_dat->active_high ^ (value == LED_OFF))
		led_bmp |= led_dat->bit;
	else
		led_bmp &= ~led_dat->bit;

}

static enum led_brightness shiftled_led_get(struct led_classdev *led_cdev)
{
	struct shiftled_led_data *led_dat =
		container_of(led_cdev, struct shiftled_led_data, cdev);

	return led_dat->brightness;
}

static int __devinit shiftled_led_probe(struct platform_device *pdev)
{
	int i;
	int err;
	struct shiftled_led_data *led;
	struct shiftled_led_platform_data *pdata = dev_get_platdata(&pdev->dev);

	led = devm_kzalloc(&pdev->dev, sizeof(struct shiftled_led_data) * pdata->num_leds,
	                   GFP_KERNEL);
	if (!led)
		return -ENOMEM;

	err = gpio_request(pdata->reg_rck, "rck");
	if (err)
		goto free_sr_data;

	err = gpio_request(pdata->reg_clk, "clk");
	if (err)
		goto free_sr_rck;

	err = gpio_request(pdata->reg_data, "data");
	if (err)
		goto free_sr_clk;

	gpio_direction_output(pdata->reg_rck, 1);
	gpio_direction_output(pdata->reg_clk, 1);
	gpio_direction_output(pdata->reg_data, 1);

	dev_set_drvdata(&pdev->dev, led);
	for(i = 0; i < pdata->num_leds; i++) {
		led[i].cdev.name = pdata->leds[i].name;
		led[i].cdev.brightness_set = shiftled_led_set;
		led[i].cdev.brightness_get = shiftled_led_get;
		led[i].cdev.default_trigger = pdata->leds[i].default_trigger;

		led[i].bit = 1 << pdata->leds[i].bit;
		led[i].active_high = pdata->leds[i].active_high;
		led[i].brightness = LED_OFF;

		if (led[i].active_high == 1)
			led_bmp &= ~led[i].bit;
		else
			led_bmp |= led[i].bit;
	}

	gpio_set_value(pdata->reg_clk, 0);
	gpio_set_value(pdata->reg_rck, 0);

	INIT_DELAYED_WORK(&shift_work, led_load_shift_reg);

	shift_pdata = pdata;
	
	led_load_shift_reg(&(shift_work.work));
	
	
	for(i = 0; i < pdata->num_leds; i++) {
		err = led_classdev_register(&pdev->dev, &led[i].cdev);
		if (err)
			goto free_leds;
	}
	return 0;

free_leds:
	while( --i >= 0)
		led_classdev_unregister(&led[i].cdev);

free_sr_data:
	gpio_free(pdata->reg_data);

free_sr_clk:
	gpio_free(pdata->reg_clk);

free_sr_rck:
	gpio_free(pdata->reg_rck);

	devm_kfree(&pdev->dev, led);
	return err;
}

static int __devexit shiftled_led_remove(struct platform_device *pdev)
{
	struct shiftled_led_platform_data *pdata = dev_get_platdata(&pdev->dev);
	struct shiftled_led_data *led = dev_get_drvdata(&pdev->dev);
	int i;

	for(i = 0; i < pdata->num_leds; i++) {
		if (led[i].cdev.name)
			led_classdev_unregister(&led[i].cdev);
	}
	devm_kfree(&pdev->dev, led);
	gpio_free(pdata->reg_data);
	gpio_free(pdata->reg_clk);
	gpio_free(pdata->reg_rck);
	return 0;
}

static struct platform_driver shiftled_driver = {
	.probe = shiftled_led_probe,
	.remove = __devexit_p(shiftled_led_remove),
	.driver = {
		.name = "shiftled-led",
		.owner = THIS_MODULE,
	},
};


static void aled_release(struct device *dev)
{
}

static struct platform_device aggregled_device = {
       .name                   = "aggreg-led",
       .id                     = 0,
       .dev = {
		.release = aled_release,
	}
};

static void aggreg_led_set(struct led_classdev *led_cdev,
                             enum led_brightness value)
{
	struct aggreg_led_data *led_dat =
		container_of(led_cdev, struct aggreg_led_data, cdev);

	led_dat->brightness = value;
	if (led_dat->led1) {
		led_set_brightness(led_dat->led1, value);
	}
	if (led_dat->led2) {
		led_set_brightness(led_dat->led2, value);
	}
}

static enum led_brightness aggreg_led_get(struct led_classdev *led_cdev)
{
	struct aggreg_led_data *led_dat =
		container_of(led_cdev, struct aggreg_led_data, cdev);

	return led_dat->brightness;
}

static int __devinit aggreg_led_probe(struct platform_device *pdev)
{
	int i;
	int err;
	struct led_classdev *led_cdev;
	struct aggreg_led_data *led;
	struct aggreg_led_platform_data *pdata = dev_get_platdata(&pdev->dev);

	led = devm_kzalloc(&pdev->dev, sizeof(struct aggreg_led_data) * pdata->num_leds,
	                   GFP_KERNEL);
	if (!led)
		return -ENOMEM;

	dev_set_drvdata(&pdev->dev, led);

	for(i = 0; i < pdata->num_leds; i++) {
		led[i].cdev.name = pdata->leds[i].name;
		led[i].cdev.brightness_set = aggreg_led_set;
		led[i].cdev.brightness_get = aggreg_led_get;
		led[i].cdev.default_trigger = pdata->leds[i].default_trigger;

		led[i].brightness = LED_OFF;

         /* Find the LED with the correct name */
         down_read(&leds_list_lock);
         list_for_each_entry(led_cdev, &leds_list, node) {
                 down_write(&led_cdev->trigger_lock);
                 if (!strcmp(pdata->leds[i].led1, led_cdev->name)) {
                     led[i].led1 = led_cdev;
                 }
                 if (!strcmp(pdata->leds[i].led2, led_cdev->name)) {
                     led[i].led2 = led_cdev;
                 }
                 up_write(&led_cdev->trigger_lock);
         }
         up_read(&leds_list_lock);

         if(!led[i].led1) {
            printk(KERN_WARNING "Could not find led1 %s\n", pdata->leds[i].led1);
         }
         if(!led[i].led2) {
            printk(KERN_WARNING "Could not find led2 %s\n", pdata->leds[i].led2);
         }
	}

	for(i = 0; i < pdata->num_leds; i++) {
		err = led_classdev_register(&pdev->dev, &led[i].cdev);
		if (err)
			goto free_leds;
	}
	return 0;

free_leds:
	while( --i >= 0)
		led_classdev_unregister(&led[i].cdev);

	devm_kfree(&pdev->dev, led);
	return err;
}

static int __devexit aggreg_led_remove(struct platform_device *pdev)
{
	struct aggreg_led_platform_data *pdata = dev_get_platdata(&pdev->dev);
	struct aggreg_led_data *led = dev_get_drvdata(&pdev->dev);
	int i;

	for(i = 0; i < pdata->num_leds; i++) {
		if (led[i].cdev.name)
			led_classdev_unregister(&led[i].cdev);
	}
	devm_kfree(&pdev->dev, led);
	return 0;
}

static struct platform_driver aggregled_driver = {
	.probe = aggreg_led_probe,
	.remove = __devexit_p(aggreg_led_remove),
	.driver = {
		.name = "aggreg-led",
		.owner = THIS_MODULE,
	},
};

/* Prototypes. */
static int __init led_init(void);
static void __exit led_exit(void);

/***************************************************************************
 * Function Name: led_init
 * Description  : Initial function that is called when the module is loaded
 * Returns      : None.
 ***************************************************************************/
static int __init led_init(void)
{
	int err;
	/* The build process for these boards does not yet fill in tch_board */
#if defined(BOARD_GANTN)
	const struct board * board_desc = get_board_description("GANT-N");
#elif defined(BOARD_C2KEVM)
	const struct board * board_desc = get_board_description("C2KEVM");
#elif defined (BOARD_GANTV)
	const struct board * board_desc = get_board_description("GANT-V");
#elif defined (BOARD_GANTW)
	const struct board * board_desc = get_board_description("GANT-W");
#elif defined (BOARD_GANT5)
	const struct board * board_desc = get_board_description("GANT-5");
#elif defined (BCM_LED_FW)
	const struct board * board_desc = bcm_led_get_board_description();
#else
	const struct board * board_desc = get_board_description(tch_board);
#endif
	struct gpio_led_platform_data *gpled_data;
	struct shiftled_led_platform_data *pdata;
	struct aggreg_led_platform_data *adata;

	shift_work_queue = create_singlethread_workqueue( "led_shift_queue" );

	if (!board_desc) {
		printk("Could not find led description for platform: %s\n", tch_board);
		return -1;
	}
	gpled_device.dev.platform_data = board_desc->gpioleds;
	shiftled_device.dev.platform_data = board_desc->shiftleds;
	aggregled_device.dev.platform_data = board_desc->aggregleds;

	gpled_data = dev_get_platdata(&gpled_device.dev);
	pdata = dev_get_platdata(&shiftled_device.dev);
	adata = dev_get_platdata(&aggregled_device.dev);

	do {
		if(gpled_data && gpled_data->num_leds > 0) {
			err = platform_device_register(&gpled_device);
			if (err) {
				printk("Failed to register GPIO led device\n");
				break;
			}
		}

		if (pdata && pdata->num_leds) {
			err = platform_driver_register(&shiftled_driver);
			if (err) {
				printk("Failed to register shiftled driver\n");
				break;
			}

			err = platform_device_register(&shiftled_device);
			if (err) {
				printk("Failed to register shiftled device\n");
				break;
			}
		}

#if defined(BCM_LED_FW)
		err = bcmled_driver_init();
		if (err) {
			break;
		}
#endif

		if (adata && adata->num_leds) {
			err = platform_driver_register(&aggregled_driver);
			if (err) {
				printk("Failed to register aggregated led driver\n");
				break;
			}

			err = platform_device_register(&aggregled_device);
			if (err) {
				printk("Failed to register aggregated led device\n");
				break;
			}
		}
	} while (0);

	return err;
}

/***************************************************************************
 * Function Name: led_exit
 * Description  : Final function that is called when the module is unloaded
 * Returns      : None.
 ***************************************************************************/
static void __exit led_exit()
{
	struct shiftled_led_platform_data *pdata = dev_get_platdata(&shiftled_device.dev);
	struct aggreg_led_platform_data *adata = dev_get_platdata(&aggregled_device.dev);


	if (adata && adata->num_leds) {
		platform_device_unregister(&aggregled_device);
		platform_driver_unregister(&aggregled_driver);
	}

	platform_device_unregister(&gpled_device);

	if (pdata && pdata->num_leds) {
		platform_device_unregister(&shiftled_device);
		platform_driver_unregister(&shiftled_driver);
	}

	destroy_workqueue( shift_work_queue );

#if defined(BCM_LED_FW)
	bcmled_driver_deinit();
#endif
}

module_init(led_init);
module_exit(led_exit);

MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("Generic LED support for Technicolor Linux Gateways");
MODULE_AUTHOR("Technicolor <linuxgw@technicolor.com");
