/************** COPYRIGHT AND CONFIDENTIALITY INFORMATION *********************
**                                                                          **
** Copyright (c) 2012 Technicolor                                           **
** All Rights Reserved                                                      **
**                                                                          **
** This program contains proprietary information which is a trade           **
** secret of TECHNICOLOR and/or its affiliates and also is protected as     **
** an unpublished work under applicable Copyright laws. Recipient is        **
** to retain this program in confidence and is not permitted to use or      **
** make copies thereof other than as permitted in a written agreement       **
** with TECHNICOLOR, UNLESS OTHERWISE EXPRESSLY ALLOWED BY APPLICABLE LAWS. **
**                                                                          **
******************************************************************************/

/* This file contains the code that doesn't depend on kernel functions and
   can therefor be compiled in userspace to run inside a unittest */

#include "rip2.h"
#include "rip2_common.h"
#include "rip_ids.h"

#define PROC_BLOCK_SIZE    (PAGE_SIZE - 1024) //taken from fs/proc/generic.c

/**
 * Handler for read system call on the RIP proc filesystem entries.
 * \param page The output buffer for the read operation
 * \param start (out) return the beginning of the data
 * \param offset The current position in the file
 * \param count The number of bytes to return (if possible)
 * \param eof We will write a "1" here to indicate EOF.
 * \param data The RIP2_ID corresponding to the file entry is stored in this
 * field
 * \return length of the data that was copied to the output buffer
 * \return -EFAULT if the read operation failed
 *
 * see fs/proc/generic.c in the kernel source code for more information
 */
int rip_proc_read(char  *page,
                  char  **start,
                  off_t offset,
                  int   count,
                  int   *eof,
                  void  *data)
{
	T_RIP2_ID   id = *(T_RIP2_ID *)data;
	int         rv;

	unsigned long length;

	/* page is at least PROC_BLOC_SIZE in length */
	length  = PROC_BLOCK_SIZE;
	rv      = rip2_drv_read(&length, id, page);
	if (rv != RIP2_SUCCESS) {
		/* no data */
		return -EFAULT;
	}

#ifndef CONFIG_RIPDRV_ANVIL
	/* append a \n if possible
	 * For Anvil it complicates things a lot, so we do not do it.
	 */
	if (length < PROC_BLOCK_SIZE) {
		page[length] = '\n';
		length++;
	}
#endif

	if (offset < length) {
		/* if all available bytes beyond offset were requested we reached eof */
		*eof = offset + count >= length;

		/* the complete item is now stored in page so the requested data starts
		 * at offset */
		*start = page + offset;

		/* return the actual number of bytes available. This can be more than
		 * requested.
		 */
		return length - offset;
	}
	else {
		/* data requested beyond EOF */
		*eof = 1; /* not strictly needed as we return 0, but it never hurts
		           * and we check for eof set in the unit test.  */
		return 0;
	}
}

