/*
 * csr8811.c -- CSR8811 BlueCore: Bluetooth ALSA SoC Audio Codec driver
 *
 * Copyright (C) 2011 DSPG (DSP Group)
 *
 * Author: Avi Miller <Avi.Miller@dspg.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

/* Note:
 * This is a "stub" asoc codec-driver, actually having no interaction with the
 * CSR8811 codec chip at all! It only allows registration of the codec within
 * the ALSA SoC framework.
 * The actual chip configuration is done by the BlueZ - Linux Bluetooth protocol stack -
 * at the user-space, via HCI commands, sent to the chip over UART.
 */
 
//#define DEBUG

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/i2c.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <sound/initval.h>
#include <sound/tlv.h>
#include <asm/div64.h>

#include "csr8811.h"

/*****************************************************************************/

#ifdef DEBUG
	#define dbg_prt(fmt, ...) \
		pr_debug("CSR asoc Cdec-drv: %s(): " fmt, __func__, ##__VA_ARGS__)

	#define DIR_STR(stream_dir)  ((stream_dir == 0) ? "for Playback" : "for Capture")
#else
	#define dbg_prt(fmt, ...)
	#define DIR_SRT(dir)
#endif

/*****************************************************************************/

#define CSR8811_FORMATS (SNDRV_PCM_FMTBIT_S8 | SNDRV_PCM_FMTBIT_S16_LE)

#define CSR8811_RATES	(SNDRV_PCM_RATE_8000 | SNDRV_PCM_RATE_48000)

/*****************************************************************************/

static struct snd_soc_codec *g_csr8811_codec;

static uint g_is_codec_master = 1;

/* codec private data */
struct csr8811_priv {
	struct snd_soc_codec codec;
};

/*****************************************************************************/

/* ASoC Controls */
static const struct snd_kcontrol_new csr8811_snd_controls[] = {
};

/* ASoC DAPM (Dynamic Audio Power Management) Widgets */
static const struct snd_soc_dapm_widget csr8811_dapm_widgets[] = {
};

/* ASoC audio-route Map */
static const struct snd_soc_dapm_route audio_route[] = {
};

/*****************************************************************************/

/* CSR8811 Codec-DAI Operations: */

static int csr8811_dai_set_clkdiv(struct snd_soc_dai *codec_dai, int div_id, int div)
{
	//struct snd_soc_codec *codec = codec_dai->codec;
	//struct csr8811_priv *csr8811 = snd_soc_codec_get_drvdata(codec);

	dbg_prt("%s\n", "Doing nothing ---->");

	return 0;
}

static int csr8811_dai_set_sysclk(struct snd_soc_dai *codec_dai, int clk_id,
				  unsigned int freq, int dir )
{
	//struct snd_soc_codec *codec = codec_dai->codec;
	//struct csr8811_priv *csr8811 = snd_soc_codec_get_drvdata(codec);

	dbg_prt("%s\n", "Doing nothing ---->");
	
	return 0;
}

static int csr8811_dai_set_pll(struct snd_soc_dai *codec_dai, int pll_id, int source,
			       unsigned int freq_in, unsigned int freq_out)
{
	dbg_prt("%s\n", "Doing nothing ---->");
	
	return 0;
}

static int csr8811_dai_set_fmt(struct snd_soc_dai *codec_dai, unsigned int fmt)
{
	//struct snd_soc_codec *codec = codec_dai->codec;
	
	dbg_prt();

	/* Set Master/Slave */
	switch (fmt & SND_SOC_DAIFMT_MASTER_MASK) {
	case SND_SOC_DAIFMT_CBM_CFM:
		dbg_prt("%s\n", "CSR is TDM Master");
		g_is_codec_master = 1;
		break;
	case SND_SOC_DAIFMT_CBS_CFS:
		dbg_prt("%s\n", "CSR is TDM Slave");
		g_is_codec_master = 0;
		break;
	default:
		return -EINVAL;
	}

	/* Set Format */
	switch (fmt & SND_SOC_DAIFMT_FORMAT_MASK) {
	case SND_SOC_DAIFMT_I2S:
		dbg_prt("%s\n", "TDM format is I2S");
		break;
	case SND_SOC_DAIFMT_RIGHT_J:
		dbg_prt("%s\n", "TDM format is Right-Justified");
		break;
	case SND_SOC_DAIFMT_LEFT_J:
		dbg_prt("%s\n", "TDM format is Left-Justified");
		break;
	default:
		return -EINVAL;
	}

	/* Set TDM clocks Polarity */
	switch (fmt & SND_SOC_DAIFMT_INV_MASK) {
	case SND_SOC_DAIFMT_NB_NF:
		dbg_prt("%s\n", "TDM clocks polarity: normal BCLK(sclk); normal LRC(FSync)");
		break;
 	case SND_SOC_DAIFMT_IB_IF:
		dbg_prt("%s\n", "TDM clocks polarity: inverted BCLK(sclk); inverted LRC(FSync)");
		break;
	case SND_SOC_DAIFMT_IB_NF:
		dbg_prt("%s\n", "TDM clocks polarity: inverted BCLK(sclk); normal LRC(FSync)");
		break;
	case SND_SOC_DAIFMT_NB_IF:
		dbg_prt("%s\n", "TDM clocks polarity: normal BCLK(sclk); inverted LRC(FSync)");
		break;
	default:
		return -EINVAL;
	}

	return 0;
}

static int csr8811_dai_hw_params(struct snd_pcm_substream *substream,
				 struct snd_pcm_hw_params *params,
				 struct snd_soc_dai *dai)
{
	//struct snd_soc_pcm_runtime *rtd = substream->private_data;
	//struct snd_soc_device *socdev = rtd->socdev;
	//struct snd_soc_codec *codec = socdev->card->codec;
	//struct csr8811_priv *csr8811 = snd_soc_codec_get_drvdata(codec);

	dbg_prt("%s\n", DIR_STR(substream->stream));
	
	/* Bit width */
	switch (params_format(params)) {
	case SNDRV_PCM_FORMAT_S8:
		dbg_prt("%s\n", "PCM sample-length (vs. ALSA) = 8 bits");
		break;
	case SNDRV_PCM_FORMAT_S16_LE:
		dbg_prt("%s\n", "PCM sample-length (vs. ALSA) = 16 bits; LE");
		break;
	default:
		return -EINVAL;
	}

	/* Filter coefficient (per sample-rate) */
	switch (params_rate(params)) {
	case 8000:
		dbg_prt("%s\n", "fs = 8KHz");
		break;
	case 48000:
		dbg_prt("%s\n", "fs = 48KHz");
		break;					
		
	case 11025:
	case 12000:
	case 16000:
	case 22050:
	case 24000:
	case 32000:
	case 44100:
	default:
		dbg_prt("%s %u %s\n",
			 "Unsupported fs:", (params_rate(params)) / 1000, "KHz");
		return -EINVAL;
	}

	return 0;
}

static int csr8811_dai_hw_free(struct snd_pcm_substream *substream,
			       struct snd_soc_dai *dai)
{
	dbg_prt("%s\n", DIR_STR(substream->stream));
	
	dbg_prt("%s\n", "Doing nothing ---->");

	return 0;
}

static int csr8811_dai_prepare(struct snd_pcm_substream *substream,
			       struct snd_soc_dai *dai)
{
	dbg_prt("%s\n", DIR_STR(substream->stream));
	
	dbg_prt("%s\n", "Doing nothing ---->");

	return 0;
}	

static int csr8811_dai_trigger(struct snd_pcm_substream *substream, int cmd,
			       struct snd_soc_dai *dai)
{
	dbg_prt("%s\n", DIR_STR(substream->stream));

	dbg_prt("%s\n", "Doing nothing ---->");

	return 0;
}	

static int csr8811_dai_mute(struct snd_soc_dai *dai, int mute)
{
	//struct snd_soc_codec *codec = dai->codec;

	dbg_prt("%s %s\n", "set mute", (mute == 0) ? "Off" : "On");

	dbg_prt("%s\n", "Doing nothing ---->");

	return 0;
}

static int csr8811_dai_startup(struct snd_pcm_substream *substream,
			       struct snd_soc_dai *dai)
{
	dbg_prt();
		
	dbg_prt("%s\n", "Doing nothing ---->");

	return 0;
}

static void csr8811_dai_shutdown(struct snd_pcm_substream *substream,
				 struct snd_soc_dai *dai)
{
	dbg_prt();

	dbg_prt("%s\n", "Doing nothing ---->");
}	
	
static struct snd_soc_dai_ops csr8811_dai_ops = {
	.set_sysclk		= csr8811_dai_set_sysclk,
	.set_pll		= csr8811_dai_set_pll,
	.set_clkdiv		= csr8811_dai_set_clkdiv,
	.set_fmt		= csr8811_dai_set_fmt,
	.digital_mute		= csr8811_dai_mute,
	.startup		= csr8811_dai_startup,
	.shutdown		= csr8811_dai_shutdown,
	.hw_params		= csr8811_dai_hw_params,
	.hw_free		= csr8811_dai_hw_free,
	.prepare		= csr8811_dai_prepare,
	.trigger		= csr8811_dai_trigger,
};

/*****************************************************************************/

/* CSR8811 Codec DAI: */

struct snd_soc_dai_driver csr8811_dai = {
	.name = "csr8811-hifi",
	.playback = {
		.stream_name	= "Playback",
		.channels_min	= 1,
		.channels_max	= 2,
		.rates 		= CSR8811_RATES,
		.formats 	= CSR8811_FORMATS,
	},
	.capture = {
		.stream_name	= "Capture",
		.channels_min	= 1,
		.channels_max	= 2,
		.rates		= CSR8811_RATES,
		.formats	= CSR8811_FORMATS,
	},
	.ops = &csr8811_dai_ops,
};

/*****************************************************************************/

/* CSR8811 Codec device: */

static int csr8811_set_bias_level(struct snd_soc_codec *codec,
				  enum snd_soc_bias_level level)
{
	dbg_prt();

	switch (level) {
	case SND_SOC_BIAS_ON:
		dbg_prt("%s\n", "Bias ON");
		break;
	case SND_SOC_BIAS_PREPARE:
		dbg_prt("%s\n", "Bias PREPARE");
		break;
	case SND_SOC_BIAS_STANDBY:
		dbg_prt("%s\n", "Bias STANDBY");
		break;
	case SND_SOC_BIAS_OFF:
		dbg_prt("%s\n", "Bias OFF");
		break;
	default:
		dbg_prt("%s %u\n", "Unsupported Bias level:", level);
		return -EINVAL;
	}

	/* Change to new state */
	codec->dapm.bias_level = level;

	return 0;
}

/*****************************************************************************/

static int csr8811_dev_probe(struct snd_soc_codec *codec)
{
	//struct csr8811_priv * csr8811 = snd_soc_codec_get_drvdata(codec);
	struct snd_soc_dapm_context *dapm = &codec->dapm;
	int ret = 0;

	dbg_prt();

	codec->control_data = NULL;

	csr8811_set_bias_level(codec, SND_SOC_BIAS_STANDBY);

	/* Register controls */
	ret = snd_soc_add_controls(codec, csr8811_snd_controls,
					ARRAY_SIZE(csr8811_snd_controls));
	if (ret < 0) {
		dev_err(codec->dev, "failed to add Controls: %d\n", ret);
		return ret;
	}

	/* Register the DAPM Widgets */
	ret = snd_soc_dapm_new_controls(dapm, csr8811_dapm_widgets,
				  ARRAY_SIZE(csr8811_dapm_widgets));
	if (ret < 0) {
		dev_err(codec->dev, "failed to add Widgets: %d\n", ret);
		return ret;
	}

	/* Set up the CSR8811 audio map */
	ret = snd_soc_dapm_add_routes(dapm, audio_route, ARRAY_SIZE(audio_route));
	if (ret < 0) {
		dev_err(codec->dev, "failed to add Routes: %d\n", ret);
		return ret;
	}
	
	return 0;
}

static int csr8811_dev_remove(struct snd_soc_codec *codec)
{
	dbg_prt();

	csr8811_set_bias_level(codec, SND_SOC_BIAS_OFF);

	return 0;
}

static int csr8811_dev_suspend(struct snd_soc_codec *codec, pm_message_t state)
{
	dbg_prt();

	csr8811_set_bias_level(codec, SND_SOC_BIAS_OFF);

	return 0;
}

static int csr8811_dev_resume(struct snd_soc_codec *codec)
{
	dbg_prt();

	csr8811_set_bias_level(codec, SND_SOC_BIAS_STANDBY);

	return 0;
}

static struct snd_soc_codec_driver soc_codec_dev_csr8811 = {
	.probe   = csr8811_dev_probe,
	.remove  = csr8811_dev_remove,
	.suspend = csr8811_dev_suspend,
	.resume  = csr8811_dev_resume,
	.set_bias_level	= csr8811_set_bias_level,
	
	/* No codec registers ! */
	.reg_cache_size	= 0,
	.reg_word_size = 0,
	.reg_cache_default = NULL,

};

/*****************************************************************************/

/* CSR8811 Codec Driver:
 * --------------------
 */

static __devinit int csr8811_register(struct csr8811_priv *csr8811)
{
	int ret;
	struct snd_soc_codec *codec = &(csr8811->codec);

	dbg_prt();

	if (g_csr8811_codec != NULL) {
		dev_err(codec->dev, "Another CSR8811 is registered. Aborting!\n");
		return -EINVAL;
	}

	codec->name = "csr8811";
	snd_soc_codec_set_drvdata(codec, csr8811);
	
	/* Register the codec */
	ret = snd_soc_register_codec(codec->dev, &soc_codec_dev_csr8811,
				     &csr8811_dai, 1);
	if (ret != 0) {
		dev_err(codec->dev, "Failed to register Codec and its Dai: %d\n", ret);
		return ret;
	}

	/* Keep global pointer to codec structure (!) */
	g_csr8811_codec = codec;

	return 0;
}

static __devexit void csr8811_unregister(struct csr8811_priv *csr8811)
{
	struct snd_soc_codec *codec = &(csr8811->codec);;

	dbg_prt();

	

	csr8811_set_bias_level(codec, SND_SOC_BIAS_OFF);
	//csr8811_set_bias_level(&csr8811->codec, SND_SOC_BIAS_OFF);

	/* Deregister the codec and its dai */
	//snd_soc_unregister_codec(csr8811->codec.dev);
	snd_soc_unregister_codec(codec->dev);

	g_csr8811_codec = NULL;
}

/*****************************************************************************/

static __devinit int csr8811_platform_probe(struct platform_device *pdev)
{
	struct csr8811_priv *csr8811 = NULL;
	struct snd_soc_codec *codec = NULL;

	dbg_prt();

	/* Allocate private data */
	csr8811 = kzalloc(sizeof(struct csr8811_priv), GFP_KERNEL);
	if (csr8811 == NULL) {
		return -ENOMEM;
	}

	/* Save pointer to private data on the device */	
	pdev->dev.platform_data = csr8811;

	/* Save pointer to device on the priv data */	
	codec = &(csr8811->codec);
	codec->dev = &(pdev->dev);
	//csr8811->codec.dev = &(pdev->dev);
	
	/* Register the codec and its dai */
	return csr8811_register(csr8811);
}


static int __devexit csr8811_platform_remove(struct platform_device *pdev)
{
	struct csr8811_priv *csr8811 = pdev->dev.platform_data;

	dbg_prt();

	/* Deregister the codec and its dai */
	csr8811_unregister(csr8811);

	kfree(csr8811);

	return 0;
}

static struct platform_driver csr8811_platform_driver = {
	.driver = {
		.name	= "BT-csr8811",
		.owner	= THIS_MODULE,
	},
	.probe	= csr8811_platform_probe,
	.remove	= __devexit_p(csr8811_platform_remove),
	.shutdown = NULL,
	.suspend  = NULL,
	.resume   = NULL,
};

static int __init csr8811_mod_Init(void)
{ 
	dbg_prt();

	return platform_driver_register(&csr8811_platform_driver);
} 
module_init(csr8811_mod_Init);

static void __exit csr8811_mod_Exit(void)
{
	dbg_prt();
	
	platform_driver_unregister(&csr8811_platform_driver);
}
module_exit(csr8811_mod_Exit);

MODULE_DESCRIPTION("ASoC CSR8811 codec driver");
MODULE_AUTHOR("Avi Miller <Avi.Miller@dspg.com>");
MODULE_LICENSE("GPL");

