/*
 * linux/drivers/video/adv7525_hdmi.c
 * I2C driver for ADV7525 (Analog Devices) HDMI controller
 *
 * Copyright (C) 2011 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/interrupt.h>
#include <linux/switch.h>
#include <linux/gpio.h>
#include <linux/mutex.h>
#include <linux/slab.h>
#include <linux/earlysuspend.h>
#include <linux/debugfs.h>
#include <linux/i2c/adv7525_hdmi.h>

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <sound/adv7525_hdmi_snd.h>

//#define DEBUG

#ifdef DEBUG
	#define dbg_prt(fmt, ...) \
		printk(KERN_DEBUG "HDMI Codec-drv: %s(): " fmt "\n", __func__, ##__VA_ARGS__)

	#define DIR_STR(stream_dir)  ((stream_dir == 0) ? "for Playback" : "for Capture")
#else
	#define dbg_prt(fmt, ...)
	#define DIR_SRT(dir)
#endif

#define HDMI_FORMATS	(SNDRV_PCM_FMTBIT_S16_LE)

#define HDMI_RATES	(SNDRV_PCM_RATE_32000	|	\
			 SNDRV_PCM_RATE_44100	|	\
			 SNDRV_PCM_RATE_48000	|	\
			 SNDRV_PCM_RATE_88200	|	\
			 SNDRV_PCM_RATE_96000	|	\
			 SNDRV_PCM_RATE_176400	|	\
			 SNDRV_PCM_RATE_192000)

/* macro format: 0xaabbcc
 *   aa: register address (offset from chip's base address)
 *   bb: field mask (8 bits mask, starting at bit0)
 *   cc: mask's right-most '1' bit-number (0 for bit0 ... 7 for bit7)
 *
 * Using this format, field Value ('val') is provided right-justified
 * (as if starting at bit 0).
 */
#define ADV7525_REG_CHIP_REVISION                   0x00ff00
#define ADV7525_REG_CTS_SELECT                      0x0a8007
#define ADV7525_REG_MCLK_I2S                        0x0a0402
#define ADV7525_REG_AUDIO_SELECT                    0x0a1004
#define ADV7525_REG_MCLK_RATIO                      0x0a0300
#define ADV7525_REG_AUDIO_CLOCK_POLARITY            0x0b4006
#define ADV7525_REG_SPDIF_ENABLE                    0x0b8007
#define ADV7525_REG_SAMPLE_FREQ_SRC_SELECT          0x0c8007
#define ADV7525_REG_I2S_FORMAT                      0x0c0300
#define ADV7525_REG_I2S_ENABLE                      0x0c3c02
#define ADV7525_REG_I2S_BIT_WIDTH                   0x0d1f00
#define ADV7525_REG_SUB_PCKT_0_L_SRC                0x0e3803
#define ADV7525_REG_SUB_PCKT_0_R_SRC                0x0e0700
#define ADV7525_REG_SUB_PCKT_1_L_SRC                0x0f3803
#define ADV7525_REG_SUB_PCKT_1_R_SRC                0x0f0700
#define ADV7525_REG_SUB_PCKT_2_L_SRC                0x103803
#define ADV7525_REG_SUB_PCKT_2_R_SRC                0x100700
#define ADV7525_REG_SUB_PCKT_3_L_SRC                0x113803
#define ADV7525_REG_SUB_PCKT_3_R_SRC                0x110700
#define ADV7525_REG_AUDIO_SAMPLE_WORD               0x128007
#define ADV7525_REG_CONSUMER_USE                    0x124006
#define ADV7525_REG_COPYRIGHT_BIT                   0x122005
#define ADV7525_REG_ADDITIONAL_AUDIO_INFO           0x121c02
#define ADV7525_REG_AUDIO_CLK_ACCURACY              0x120300
#define ADV7525_REG_CATEGORY_CODE                   0x13ff00
#define ADV7525_REG_SOURCE_NUMBER                   0x14F004
#define ADV7525_REG_WORD_LENGTH                     0x140F00
#define ADV7525_REG_SPEAKER_MAPPING                 0x76ff00
#define ADV7525_REG_INPUT_VIDEO_FORMAT              0x150e01
#define ADV7525_REG_SAMPLING_FREQUENCY              0x15f004
#define ADV7525_REG_ASPECT_RATIO                    0x170201
#define ADV7525_REG_POWER_DOWN                      0x414006
#define ADV7525_REG_RX_SENSE_STATE                  0x422005
#define ADV7525_REG_CHANNEL_COUNT                   0x730700
#define ADV7525_REG_RX_SENSE_INTERRUPT_ENABLE       0x944006
#define ADV7525_REG_HPD_INTERRUPT_ENABLE            0x948007
#define ADV7525_REG_RX_SENSE_INTERRUPT              0x964006
#define ADV7525_REG_HPD_INTERRUPT                   0x968007
#define ADV7525_REG_RX_SENSE_POWER_DOWN             0xa14006
#define ADV7525_REG_HDMI_MODE                       0xaf0201
#define ADV7525_REG_CLOCK_DELAY                     0xbae005
#define ADV7525_REG_VIDEO_INPUT_AND_CLOCK_GATING    0xd60100
#define ADV7525_REG_MCLK_MUX                        0xd64006
#define ADV7525_REG_CEC_POWER_DOWN                  0xe20100
#define ADV7525_REG_HDCP_CONTROLLER_POWER           0xe40201
#define ADV7525_REG_V1P2_ENABLE                     0xe48007
#define ADV7525_REG_HSYNC_MCLK_SCHMITT_TRIGGER      0xe50803
#define ADV7525_REG_RX_SENSE_MONITOR_OSC_POWER_DOWN 0xe60100
#define ADV7525_REG_CEC_CLOCK_INPUT_ENABLE          0xe62005

struct adv7525_dev {
	struct i2c_client *client;
	struct switch_dev hdmi_sdev;
	struct work_struct sense_work;
	struct adv7525_platform_data *pdata;
	struct early_suspend early_suspend;
	struct attribute_group *attr_group;
	struct mutex mutex;
	struct dentry *dentry;
	int mode;
	int irq;
};

enum adv7525_mode {
	ADV7525_MODE_OFF,
	ADV7525_MODE_MONITOR,
	ADV7525_MODE_ON,
};

static unsigned char identity_matrix[24] = {
	0xA8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 8, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 8, 0, 0, 0
};

static int
adv7525_read(struct adv7525_dev *adv7525, int reg)
{
	return i2c_smbus_read_byte_data(adv7525->client, reg);
}

static int
adv7525_write(struct adv7525_dev *adv7525, int reg, uint8_t val)
{
	return i2c_smbus_write_byte_data(adv7525->client, reg, val);
}

static int
adv7525_write_data(struct adv7525_dev *adv7525, int reg, unsigned char *data,
		   int size)
{
	return i2c_smbus_write_block_data(adv7525->client, reg, size, data);
}

static int
adv7525_read_reg(struct adv7525_dev *adv7525, int reg)
{
	char tmp, shift, mask;

	tmp = adv7525_read(adv7525, reg >> 16);
	mask = (reg >> 8) & 0xff;
	shift = reg & 0xff;

	return (tmp & mask) >> shift;
}

static void
adv7525_write_reg(struct adv7525_dev *adv7525, int reg, int val)
{
	char tmp, shift, mask;

	tmp = adv7525_read(adv7525, reg >> 16);
	mask = (reg >> 8) & 0xff;
	shift = reg & 0xff;

	tmp = (tmp & ~mask) | (char)((val << shift) & mask);
	adv7525_write(adv7525, reg >> 16, tmp);
}

static void
adv7525_write_reg_atomic(struct adv7525_dev *adv7525, int reg, int val)
{
	mutex_lock(&adv7525->mutex);

	adv7525_write_reg(adv7525, reg, val);

	mutex_unlock(&adv7525->mutex);
}

static int
adv7525_dai_set_clkdiv(struct snd_soc_dai *dai, int div_id, int div)
{
	dbg_prt("%s", "Doing nothing ---->");

	return 0;
}

static int
adv7525_dai_set_sysclk(struct snd_soc_dai *dai, int clk_id,
                       unsigned int freq, int dir)
{
	dbg_prt("%s", "Doing nothing ---->");

	return 0;
}

static int
adv7525_dai_set_pll(struct snd_soc_dai *codec_dai, int pll_id, int source,
                    unsigned int freq_in, unsigned int freq_out)
{
	dbg_prt("%s", "Doing nothing ---->");

	return 0;
}

static int
adv7525_dai_set_fmt(struct snd_soc_dai *dai, unsigned int fmt)
{
	struct adv7525_dev *adv7525 = snd_soc_dai_get_drvdata(dai);

	dbg_prt();

	/* Set Master/Slave */
	switch (fmt & SND_SOC_DAIFMT_MASTER_MASK) {
	case SND_SOC_DAIFMT_CBM_CFM:
		dbg_prt("%s", "HDMI is TDM Master. Unsupported. Aborting!");
		return -EINVAL;
		break;
	case SND_SOC_DAIFMT_CBS_CFS:
		dbg_prt("%s", "HDMI is TDM Slave");
		break;
	default:
		dbg_prt("%s",
			"Unsupported Master/Slave setting. Aborting!");
		return -EINVAL;
	}

	/* Set Format */
	switch (fmt & SND_SOC_DAIFMT_FORMAT_MASK) {
	case SND_SOC_DAIFMT_I2S:
		dbg_prt("%s", "TDM format is I2S");
		adv7525_write_reg_atomic(adv7525, ADV7525_REG_AUDIO_SELECT, 0); /* I2S */
		adv7525_write_reg_atomic(adv7525, ADV7525_REG_MCLK_I2S, 0); /* MCLK inactive */
		adv7525_write_reg_atomic(adv7525, ADV7525_REG_I2S_FORMAT, 0); /* I2S format */
		break;
	case SND_SOC_DAIFMT_RIGHT_J:
		dbg_prt("%s",
			"TDM format is Right-Justified. Unsupported yet. Aborting!");
		return -EINVAL;
		break;
	default:
		dbg_prt("%s", "Unsupported TDM format. Aborting!");
		return -EINVAL;
	}

	/* Set TDM clocks Polarity */
	switch (fmt & SND_SOC_DAIFMT_INV_MASK) {
	case SND_SOC_DAIFMT_NB_NF:
		/* normal BCLK(sclk); normal LRC(FSync) */
	case SND_SOC_DAIFMT_NB_IF:
		/* normal BCLK(sclk); inverted LRC(FSync) */
		dbg_prt("%s", "Latching on sclk Rise");
		/* Reg 0x0B: bit 6 : '0' - RX data latched at sclk Rising edge */
		adv7525_write_reg_atomic(adv7525, ADV7525_REG_AUDIO_CLOCK_POLARITY, 0);
		break;
	case SND_SOC_DAIFMT_IB_IF:
		/* inverted BCLK(sclk); inverted LRC(FSync) */
	case SND_SOC_DAIFMT_IB_NF:
		/* inverted BCLK(sclk); normal LRC(FSync) */
		dbg_prt("%s", "Latching on sclk Fall");
		/* Reg 0x0B: bit 6 : '1' - RX data latched at sclk Falling edge */
		adv7525_write_reg_atomic(adv7525, ADV7525_REG_AUDIO_CLOCK_POLARITY, 1);
		break;
	default:
		dbg_prt("%s", "Unsupported TDM clock polarity . Aborting!");
		return -EINVAL;
	}

	return 0;
}

static int
adv7525_dai_hw_params(struct snd_pcm_substream *substream,
                      struct snd_pcm_hw_params *params,
                      struct snd_soc_dai *dai)
{
	struct adv7525_dev *adv7525 = snd_soc_dai_get_drvdata(dai);
	int reg_val;

	/* For 20-bit "N" and "CTS" values, used to regenerate the audio
	 * clock in the receiver, we assume a video clock of 74.25 MHz.
	 */
	int clk_regen_N = 0;
	int clk_regen_CTS = 0;

	dbg_prt("%s", DIR_STR(substream->stream));

	/* Bit width */
	switch (params_format(params)) {
	case SNDRV_PCM_FORMAT_S16_LE:
		dbg_prt("%s", "PCM sample-length (vs. ALSA) = 16 bits; LE");
		/* Reg 0x0D: bits 4:0 : Bit-width = 16.
		 * relevant for Right-Justified only (!), 16 - 24 bits.
		 */
		adv7525_write_reg_atomic(adv7525, ADV7525_REG_I2S_BIT_WIDTH, 16);
		break;
	default:
		dbg_prt("%s", "Unsupported PCM sample-length. Aborting");
		return -EINVAL;
	}

	/* Filter coefficient (per sample-rate) */
	switch (params_rate(params)) {
	case 32000:
		dbg_prt("%s", "fs = 32KHz");
		reg_val = 0x3;
		clk_regen_N = 4096;
		clk_regen_CTS = 74250;
		break;
	case 44100:
		dbg_prt("%s", "fs = 44.1KHz");
		reg_val = 0x0;
		clk_regen_N = 6272;
		clk_regen_CTS = 82500;
		break;
	case 48000:
		dbg_prt("%s", "fs = 48KHz");
		reg_val = 0x2;
		clk_regen_N = 6144;
		clk_regen_CTS = 74250;
		break;
	case 88200:
		dbg_prt("%s", "fs = 88.2KHz");
		reg_val = 0x8;
		clk_regen_N = 12544;
		clk_regen_CTS = 82500;
		break;
	case 96000:
		dbg_prt("%s", "fs = 96KHz");
		reg_val = 0xa;
		clk_regen_N = 12288;
		clk_regen_CTS = 74250;
		break;
	case 176400:
		dbg_prt("%s", "fs = 176.4KHz");
		reg_val = 0xc;
		clk_regen_N = 25088;
		clk_regen_CTS = 82500;
		break;
	case 192000:
		dbg_prt("%s", "fs = 192KHz");
		reg_val = 0xe;
		clk_regen_N = 24576;
		clk_regen_CTS = 74250;
		break;

	case 8000:
	case 11025:
	case 12000:
	case 16000:
	case 22050:
	case 24000:
	default:
		dbg_prt("%s %u %s",
			 "Unsupported fs:", (params_rate(params)) / 1000, "KHz. Aborting!");
		return -EINVAL;
	}
	/* Reg 0x15: bits 7:4 : Sampling freq */
	adv7525_write_reg_atomic(adv7525, ADV7525_REG_SAMPLING_FREQUENCY, reg_val);

	/* "N" value into Regs 0x01, 0x02, 0x03 */
	adv7525_write(adv7525, 0x01, (clk_regen_N >> 16) & 0x0F); /* bits 19-16 */
	adv7525_write(adv7525, 0x02, (clk_regen_N >>  8) & 0xFF); /* bits 15- 8 */
	adv7525_write(adv7525, 0x03, (clk_regen_N >>  0) & 0xFF); /* bits  7- 0 */

	/* "CTS Manual" value into Regs 0x07, 0x08, 0x09 */
	adv7525_write(adv7525, 0x07, (clk_regen_CTS >> 16) & 0x0F); /* bits 19-16 */
	adv7525_write(adv7525, 0x08, (clk_regen_CTS >>  8) & 0xFF); /* bits 15- 8 */
	adv7525_write(adv7525, 0x09, (clk_regen_CTS >>  0) & 0xFF); /* bits  7- 0 */

	/* Reg 0x0A: bit 8 : CTS Source select - Manual CTS */
	//adv7525_write_reg_atomic(adv7525, ADV7525_REG_CTS_SELECT, 1);
	/* Reg 0x0A: bit 8 : CTS Source select - Auto CTS */
	adv7525_write_reg_atomic(adv7525, ADV7525_REG_CTS_SELECT, 0);

	/* Reg 0x0A: bits 1:0 : MCLK Ratio - 128 x fs */
	adv7525_write_reg_atomic(adv7525, ADV7525_REG_MCLK_RATIO, 0);

	/* Reg Number of audio channels */
	//reg_val = params_channels(params);
	reg_val = 2; /* currently */
	if ((reg_val >= 2) && (reg_val <= 8)) {
		/* Reg 0x73: bits 2:0 : Channel-Count - 1 */
		adv7525_write_reg(adv7525, ADV7525_REG_CHANNEL_COUNT,
				  (reg_val - 1));
	} else {
		/* "Refer to Stream Header" / CEA 861 */
		adv7525_write_reg(adv7525, ADV7525_REG_CHANNEL_COUNT, 0);
	}

	/* Reg 0x0C: bit 7 : fs taken, for "I2S AES3 Direct Mode", from the i2s stream */
	adv7525_write_reg_atomic(adv7525, ADV7525_REG_SAMPLE_FREQ_SRC_SELECT, 0);
	/* Reg 0x0C: bit 7 : fs taken, for "I2S AES3 Direct Mode", from reg 0x15 [7:4] val */
	//adv7525_write_reg_atomic(adv7525, ADV7525_REG_SAMPLE_FREQ_SRC_SELECT, 1);

	adv7525_write_reg_atomic(adv7525, ADV7525_REG_WORD_LENGTH, 2); /* 16 bits */

	return 0;
}

static int
adv7525_dai_hw_free(struct snd_pcm_substream *substream,
                    struct snd_soc_dai *dai)
{
	struct adv7525_dev *adv7525 = snd_soc_dai_get_drvdata(dai);

	dbg_prt("%s", DIR_STR(substream->stream));

	/* Reg 0x0C: bits 5:2 : Disabling all TDM inputs "I2S 3/2/1/0" */
	adv7525_write_reg_atomic(adv7525, ADV7525_REG_I2S_ENABLE, 0);

	return 0;
}

static int
adv7525_dai_prepare(struct snd_pcm_substream *substream,
                    struct snd_soc_dai *dai)
{
	struct adv7525_dev *adv7525 = snd_soc_dai_get_drvdata(dai);

	dbg_prt("%s", DIR_STR(substream->stream));

	/* Reg 0x0C: bits 5:2 : Enabling TDM inputs "I2S 3/2/1/0" */
	switch (adv7525->pdata->i2s_port) {
	case 0:
		/* Enable I2S0 input */
		adv7525_write_reg_atomic(adv7525, ADV7525_REG_I2S_ENABLE, 0x1);
		/* Set subpacket 0-L from I2S0-L */
		adv7525_write_reg_atomic(adv7525, ADV7525_REG_SUB_PCKT_0_L_SRC, 0X0);
		/* Set subpacket 0-R from I2S0-R */
		adv7525_write_reg_atomic(adv7525, ADV7525_REG_SUB_PCKT_0_R_SRC, 0X1);
		break;
	case 1:
		/* Enable I2S1 input */
		adv7525_write_reg_atomic(adv7525, ADV7525_REG_I2S_ENABLE, 0x2);
		/* Set subpacket 0-L from I2S1-L */
		adv7525_write_reg_atomic(adv7525, ADV7525_REG_SUB_PCKT_0_L_SRC, 0X2);
		/* Set subpacket 0-R from I2S1-R */
		adv7525_write_reg_atomic(adv7525, ADV7525_REG_SUB_PCKT_0_R_SRC, 0X3);
		break;
	case 2:
		/* Enable I2S2 input */
		adv7525_write_reg_atomic(adv7525, ADV7525_REG_I2S_ENABLE, 0x4);
		/* Set subpacket 0-L from I2S1-L */
		adv7525_write_reg_atomic(adv7525, ADV7525_REG_SUB_PCKT_0_L_SRC, 0X4);
		/* Set subpacket 0-R from I2S1-R */
		adv7525_write_reg_atomic(adv7525, ADV7525_REG_SUB_PCKT_0_R_SRC, 0X5);
		break;
	case 3:
		/* Enable I2S3 input */
		adv7525_write_reg_atomic(adv7525, ADV7525_REG_I2S_ENABLE, 0x8);
		/* Set subpacket 0-L from I2S1-L */
		adv7525_write_reg_atomic(adv7525, ADV7525_REG_SUB_PCKT_0_L_SRC, 0X6);
		/* Set subpacket 0-R from I2S1-R */
		adv7525_write_reg_atomic(adv7525, ADV7525_REG_SUB_PCKT_0_R_SRC, 0X7);
		break;

	default:
		dbg_prt("%s", "Invalid ADV7525 I2S port");
		return -EINVAL;
	}

	return 0;
}

static int
adv7525_dai_trigger(struct snd_pcm_substream *substream, int cmd,
                    struct snd_soc_dai *dai)
{
	dbg_prt("%s", DIR_STR(substream->stream));

	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
	case SNDRV_PCM_TRIGGER_RESUME:
	case SNDRV_PCM_TRIGGER_PAUSE_RELEASE:
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
			dbg_prt("%s", "Playback Start/Resume/unPause");
		} else {
			dbg_prt("%s", "Capture Start/Resume/unPause");
		}
		break;

	case SNDRV_PCM_TRIGGER_STOP:
	case SNDRV_PCM_TRIGGER_SUSPEND:
	case SNDRV_PCM_TRIGGER_PAUSE_PUSH:
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
			dbg_prt("%s", "Playback Stop/Suspend/Pause");
		} else {
			dbg_prt("%s", "Capture Stop/Suspend/Pause");
		}
		break;

	default:
		dbg_prt("%s", "Unhandled trigger");
		return -EINVAL;
	}

	return 0;
}

static int
adv7525_dai_mute(struct snd_soc_dai *dai, int mute)
{
	dbg_prt("%s %s", "set mute", (mute == 0) ? "Off" : "On");

	dbg_prt("%s", "Doing nothing ---->");

	return 0;
}

static int
adv7525_dai_startup(struct snd_pcm_substream *substream,
                    struct snd_soc_dai *dai)
{
	dbg_prt();

	dbg_prt("%s", "Doing nothing ---->");

	return 0;
}

static void
adv7525_dai_shutdown(struct snd_pcm_substream *substream,
                     struct snd_soc_dai *dai)
{
	dbg_prt();

	dbg_prt("%s", "Doing nothing ---->");
}

static struct snd_soc_dai_ops adv7525_dai_ops = {
	.set_sysclk	= adv7525_dai_set_sysclk,
	.set_pll	= adv7525_dai_set_pll,
	.set_clkdiv	= adv7525_dai_set_clkdiv,
	.set_fmt	= adv7525_dai_set_fmt,
	.digital_mute	= adv7525_dai_mute,
	.startup	= adv7525_dai_startup,
	.shutdown	= adv7525_dai_shutdown,
	.hw_params	= adv7525_dai_hw_params,
	.hw_free	= adv7525_dai_hw_free,
	.prepare	= adv7525_dai_prepare,
	.trigger	= adv7525_dai_trigger,
};

static struct snd_soc_dai_driver adv7525_hdmi_dai = {
	.name = "adv7525-hdmi-hifi",
	.playback = {
		.stream_name	= "Playback",
		.channels_min	= 1,
		.channels_max	= 2, /* future: 6 channels */
		.rates		= HDMI_RATES,
		.formats	= HDMI_FORMATS,
	},

	/* No Capture ! */

	.ops = &adv7525_dai_ops,
};

static int
adv7525_codec_set_bias_level(struct snd_soc_codec *codec,
                             enum snd_soc_bias_level level)
{
	dbg_prt();

	switch (level) {
	case SND_SOC_BIAS_ON:
		dbg_prt("%s", "Bias ON");
		break;
	case SND_SOC_BIAS_PREPARE:
		dbg_prt("%s", "Bias PREPARE");
		break;
	case SND_SOC_BIAS_STANDBY:
		dbg_prt("%s", "Bias STANDBY");
		break;
	case SND_SOC_BIAS_OFF:
		dbg_prt("%s", "Bias OFF");
		break;
	default:
		dbg_prt("%s %u", "Unsupported Bias level:", level);
		return -EINVAL;
	}

	/* Change to new state */
	codec->dapm.bias_level = level;

	return 0;
}

/* ASoC Controls */
static const struct snd_kcontrol_new adv7525_snd_controls[] = {
};

/* ASoC DAPM (Dynamic Audio Power Management) Widgets */
static const struct snd_soc_dapm_widget adv7525_dapm_widgets[] = {
};

/* ASoC audio-route Map */
static const struct snd_soc_dapm_route adv7525_audio_route[] = {
};

static int
adv7525_codec_probe(struct snd_soc_codec *codec)
{
	struct snd_soc_dapm_context *dapm = &codec->dapm;
	int ret = 0;

	dbg_prt();

	codec->control_data = NULL;

	adv7525_codec_set_bias_level(codec, SND_SOC_BIAS_STANDBY);

	/* Register controls */
	ret = snd_soc_add_controls(codec, adv7525_snd_controls,
	                           ARRAY_SIZE(adv7525_snd_controls));
	if (ret < 0) {
		dev_err(codec->dev, "failed to add controls: %d\n", ret);
		return ret;
	}

	/* Register the DAPM Widgets */
	ret = snd_soc_dapm_new_controls(dapm, adv7525_dapm_widgets,
	                                ARRAY_SIZE(adv7525_dapm_widgets));
	if (ret < 0) {
		dev_err(codec->dev, "failed to add widgets: %d\n", ret);
		return ret;
	}

	/* Set up the HDMI audio map */
	ret = snd_soc_dapm_add_routes(dapm, adv7525_audio_route,
	                              ARRAY_SIZE(adv7525_audio_route));
	if (ret < 0) {
		dev_err(codec->dev, "failed to add routes: %d\n", ret);
		return ret;
	}

	return 0;
}

static int
adv7525_codec_remove(struct snd_soc_codec *codec)
{
	dbg_prt();

	adv7525_codec_set_bias_level(codec, SND_SOC_BIAS_OFF);

	return 0;
}

static int
adv7525_codec_suspend(struct snd_soc_codec *codec, pm_message_t state)
{
	dbg_prt();

	adv7525_codec_set_bias_level(codec, SND_SOC_BIAS_OFF);

	return 0;
}

static int
adv7525_codec_resume(struct snd_soc_codec *codec)
{
	dbg_prt();

	adv7525_codec_set_bias_level(codec, SND_SOC_BIAS_STANDBY);

	return 0;
}

static struct snd_soc_codec_driver soc_codec_dev_adv7525_hdmi = {
	.probe   = adv7525_codec_probe,
	.remove  = adv7525_codec_remove,
	.suspend = adv7525_codec_suspend,
	.resume  = adv7525_codec_resume,
	.set_bias_level = adv7525_codec_set_bias_level,

	/* No codec registers ! */
	.reg_cache_size = 0,
	.reg_word_size = 0,
	.reg_cache_default = NULL,
};

static void
adv7525_check_state(struct adv7525_dev *adv7525)
{
	if (!adv7525_read_reg(adv7525, ADV7525_REG_RX_SENSE_POWER_DOWN)) {
		if (adv7525_read_reg(adv7525, ADV7525_REG_RX_SENSE_STATE))
			switch_set_state(&adv7525->hdmi_sdev, 1);
		else
			switch_set_state(&adv7525->hdmi_sdev, 0);
	}
}

static void
adv7525_sense_work(struct work_struct *data)
{
	struct adv7525_dev *adv7525 =
		container_of(data, struct adv7525_dev, sense_work);
	int val;

	mutex_lock(&adv7525->mutex);

	val = adv7525_read_reg(adv7525, ADV7525_REG_RX_SENSE_INTERRUPT);
	if (val) {
		adv7525_check_state(adv7525);
		adv7525_write_reg(adv7525, ADV7525_REG_RX_SENSE_INTERRUPT, 1);
	}

	mutex_unlock(&adv7525->mutex);
}

static irqreturn_t
adv7525_isr(int irq, void *dev_id)
{
	struct adv7525_dev *adv7525 = (struct adv7525_dev *)dev_id;

	schedule_work(&adv7525->sense_work);

	return IRQ_HANDLED;
}

static int
adv7525_init_controller(struct adv7525_dev *adv7525)
{
	/* power up */
	adv7525_write_reg(adv7525, ADV7525_REG_POWER_DOWN, 0);
	if (adv7525_read_reg(adv7525, ADV7525_REG_POWER_DOWN))
		return -ENODEV;

	/* fixed registers */
	adv7525_write_reg(adv7525, 0x980f00, 3);
	adv7525_write_reg(adv7525, 0x9cff00, 0x38);
	adv7525_write_reg(adv7525, 0x9d0300, 1);
	adv7525_write_reg(adv7525, 0xa2ff00, 0xa0);
	adv7525_write_reg(adv7525, 0xa3ff00, 0xa0);
	adv7525_write_reg(adv7525, 0xdeff00, 0x82);
	adv7525_write_reg(adv7525, 0xe44006, 1);
	adv7525_write_reg(adv7525, 0xe40402, 1);
	adv7525_write_reg(adv7525, 0xe58007, 1);
	adv7525_write_reg(adv7525, 0xe60201, 0);
	adv7525_write_reg(adv7525, 0xeb0201, 1);

	/* no delay for input video clk capture */
	adv7525_write_reg(adv7525, ADV7525_REG_CLOCK_DELAY, 3);
	/* disable HDCP */
	adv7525_write_reg(adv7525, ADV7525_REG_HDCP_CONTROLLER_POWER, 0);
	/* select HDMI */
	adv7525_write_reg(adv7525, ADV7525_REG_HDMI_MODE, 1);

	adv7525_write_data(adv7525, 0x18, identity_matrix, 24);

	/* input video format = 24bit RGB */
	adv7525_write_reg(adv7525, ADV7525_REG_INPUT_VIDEO_FORMAT, 0);

	/* aspect ratio of input video = 16x9 */
	adv7525_write_reg(adv7525, ADV7525_REG_ASPECT_RATIO, 1);

	/* CEC power down */
	adv7525_write_reg(adv7525, ADV7525_REG_CEC_POWER_DOWN, 1);

	/* disable CEC clock */
	adv7525_write_reg(adv7525, ADV7525_REG_CEC_CLOCK_INPUT_ENABLE, 0);

	/* DVDD1P2 is 1.2V */
	adv7525_write_reg(adv7525, ADV7525_REG_V1P2_ENABLE, 1);

	/* Audio configuration */
	adv7525_write_reg(adv7525, ADV7525_REG_SPDIF_ENABLE, 0);
	adv7525_write_reg(adv7525, ADV7525_REG_CHANNEL_COUNT, 0);
	adv7525_write_reg(adv7525, ADV7525_REG_MCLK_MUX, 0);
	adv7525_write_reg(adv7525, ADV7525_REG_HSYNC_MCLK_SCHMITT_TRIGGER, 0);
	adv7525_write_reg(adv7525, ADV7525_REG_AUDIO_SAMPLE_WORD, 0);
	adv7525_write_reg(adv7525, ADV7525_REG_CONSUMER_USE, 0);
	adv7525_write_reg(adv7525, ADV7525_REG_COPYRIGHT_BIT, 0);
	adv7525_write_reg(adv7525, ADV7525_REG_ADDITIONAL_AUDIO_INFO, 0);
	adv7525_write_reg(adv7525, ADV7525_REG_AUDIO_CLK_ACCURACY, 0);
	adv7525_write_reg(adv7525, ADV7525_REG_CATEGORY_CODE, 0);
	adv7525_write_reg(adv7525, ADV7525_REG_SOURCE_NUMBER, 0);
	adv7525_write_reg(adv7525, ADV7525_REG_SPEAKER_MAPPING, 0);

	/* clear RX sense interrupt */
	adv7525_write_reg(adv7525, ADV7525_REG_RX_SENSE_INTERRUPT, 1);

	/* disable RX sense interrupt */
	adv7525_write_reg(adv7525, ADV7525_REG_RX_SENSE_INTERRUPT_ENABLE, 0);

	/* clear HPD interrupt */
	adv7525_write_reg(adv7525, ADV7525_REG_HPD_INTERRUPT, 1);

	/* disable HPD interrupt */
	adv7525_write_reg(adv7525, ADV7525_REG_HPD_INTERRUPT_ENABLE, 0);

	return 0;
}

static void
adv7525_power_control(struct adv7525_dev *adv7525, int mode)
{
	mutex_lock(&adv7525->mutex);

	if (gpio_is_valid(adv7525->pdata->power_gpio) && mode)
		gpio_set_value(adv7525->pdata->power_gpio, 1);

	if (mode)
		adv7525_write_reg(adv7525, ADV7525_REG_RX_SENSE_MONITOR_OSC_POWER_DOWN, 0);

	adv7525_write_reg(adv7525, ADV7525_REG_RX_SENSE_INTERRUPT, 1);

	adv7525_write_reg(adv7525, ADV7525_REG_RX_SENSE_INTERRUPT_ENABLE, 0);

	adv7525_write_reg(adv7525, ADV7525_REG_VIDEO_INPUT_AND_CLOCK_GATING,
			  !mode);

	adv7525_write_reg(adv7525, ADV7525_REG_POWER_DOWN,
			  !(mode==ADV7525_MODE_ON));

	adv7525_write_reg(adv7525, ADV7525_REG_RX_SENSE_POWER_DOWN, !mode);

	if (mode) {
		/* restore all relevant registers between 0xaf and 0xcc */

		/* no delay for input video clk capture */
		adv7525_write_reg(adv7525, ADV7525_REG_CLOCK_DELAY, 3);

		/* disable HDCP */
		adv7525_write_reg(adv7525, ADV7525_REG_HDCP_CONTROLLER_POWER, 0);

		/* select HDMI */
		adv7525_write_reg(adv7525, ADV7525_REG_HDMI_MODE, 1);

		/* CEC power down */
		adv7525_write_reg(adv7525, ADV7525_REG_CEC_POWER_DOWN, 1);

		/* disable CEC clock */
		adv7525_write_reg(adv7525, ADV7525_REG_CEC_CLOCK_INPUT_ENABLE, 0);

		/* DVDD1P2 is 1.2V */
		adv7525_write_reg(adv7525, ADV7525_REG_V1P2_ENABLE, 1);

		/* Audio configuration */
		adv7525_write_reg(adv7525, ADV7525_REG_MCLK_MUX, 0);
		adv7525_write_reg(adv7525, ADV7525_REG_HSYNC_MCLK_SCHMITT_TRIGGER, 0);

		adv7525_write_reg(adv7525, ADV7525_REG_RX_SENSE_INTERRUPT, 1);

		adv7525_write_reg(adv7525, ADV7525_REG_RX_SENSE_INTERRUPT_ENABLE, 1);
	} else {
		adv7525_write_reg(adv7525, ADV7525_REG_RX_SENSE_MONITOR_OSC_POWER_DOWN, 1);
	}

	if (gpio_is_valid(adv7525->pdata->power_gpio) && !mode)
		gpio_set_value(adv7525->pdata->power_gpio, 0);

	adv7525_check_state(adv7525);

	mutex_unlock(&adv7525->mutex);
}

static ssize_t
adv7525_sysfs_show_mode(struct device *dev, struct device_attribute *attr,
                        char *buf)
{
	struct switch_dev *sdev = (struct switch_dev *)dev_get_drvdata(dev);
	struct adv7525_dev *adv7525 =
		container_of(sdev, struct adv7525_dev, hdmi_sdev);
	int ret = 0;

	mutex_lock(&adv7525->mutex);

	switch (adv7525->mode) {
	case ADV7525_MODE_OFF:
		ret = scnprintf(buf, PAGE_SIZE, "off\n");
		break;
	case ADV7525_MODE_MONITOR:
		ret = scnprintf(buf, PAGE_SIZE, "monitor\n");
		break;
	case ADV7525_MODE_ON:
		ret = scnprintf(buf, PAGE_SIZE, "on\n");
		break;
	default:
		ret = scnprintf(buf, PAGE_SIZE, "invalid\n");
		break;
	}

	mutex_unlock(&adv7525->mutex);

	return ret;
}

static ssize_t
adv7525_sysfs_store_mode(struct device *dev, struct device_attribute *attr,
                         const char *buf, size_t count)
{
	struct switch_dev *sdev = (struct switch_dev *)dev_get_drvdata(dev);
	struct adv7525_dev *adv7525 =
		container_of(sdev, struct adv7525_dev, hdmi_sdev);
	int ret = count;

	if (!strncmp(buf, "on", 2)) {
		adv7525_power_control(adv7525, ADV7525_MODE_ON);
		adv7525->mode = ADV7525_MODE_ON;
	} else if (!strncmp(buf, "monitor", 7)) {
		adv7525_power_control(adv7525, ADV7525_MODE_MONITOR);
		adv7525->mode = ADV7525_MODE_MONITOR;
	} else if (!strncmp(buf, "off", 3)) {
		adv7525_power_control(adv7525, ADV7525_MODE_OFF);
		adv7525->mode = ADV7525_MODE_OFF;
	} else {
		ret = -EINVAL;
	}

	return ret;
}

static DEVICE_ATTR(mode, S_IWUSR | S_IRUGO, adv7525_sysfs_show_mode,
	adv7525_sysfs_store_mode);

static struct attribute *adv7525_attrs[] = {
	&dev_attr_mode.attr,
	NULL
};

static struct attribute_group adv7525_attr_group = {
	.attrs = adv7525_attrs,
};

#ifdef CONFIG_DEBUG_FS
static ssize_t
adv7525_debugfs_read(struct file *file, char __user *userbuf, size_t count,
                     loff_t *ppos)
{
	struct adv7525_dev *adv7525 = file->private_data;
	char *buf = kmalloc(4096, GFP_KERNEL);
	ssize_t size_read = 0;
	int len = 0;
	int i;

	if (buf) {
		mutex_lock(&adv7525->mutex);
		for (i = 0; i <= 0xFF; i++)
			len += sprintf(buf + len, "0x%02x: 0x%02x\n", i,
			                adv7525_read(adv7525, i)) + 1;
		mutex_unlock(&adv7525->mutex);

		size_read = simple_read_from_buffer(userbuf, count, ppos, buf, len);
		kfree(buf);
	}

	return size_read;
}

static int
adv7525_debugfs_open(struct inode *inode, struct file *file)
{
	file->private_data = inode->i_private;

	return 0;
}

static const struct file_operations adv7525_debugfs_operations = {
	.open		= adv7525_debugfs_open,
	.read		= adv7525_debugfs_read,
	.llseek		= default_llseek,
};
#endif

#ifdef CONFIG_HAS_EARLYSUSPEND
static void
adv7525_early_suspend(struct early_suspend *es)
{
	struct adv7525_dev *adv7525 =
		container_of(es, struct adv7525_dev, early_suspend);

	if (!adv7525->mode)
		return;

	adv7525_power_control(adv7525, ADV7525_MODE_OFF);
}

static void
adv7525_late_resume(struct early_suspend *es)
{
	struct adv7525_dev *adv7525 =
		container_of(es, struct adv7525_dev, early_suspend);

	if (adv7525->mode)
		adv7525_power_control(adv7525, adv7525->mode);
}
#endif

static int __devinit
adv7525_probe(struct i2c_client *client,
              const struct i2c_device_id *dev_id)
{
	struct adv7525_dev *adv7525;
	struct adv7525_platform_data *pdata = client->dev.platform_data;
	int err = 0;

	if (!pdata) {
		dev_err(&client->dev, "missing platform data.\n");
		return -EINVAL;
	}

	if (!i2c_check_functionality(client->adapter,
	                             I2C_FUNC_SMBUS_BYTE |
	                             I2C_FUNC_SMBUS_BYTE_DATA |
	                             I2C_FUNC_SMBUS_WRITE_BLOCK_DATA)) {
		dev_err(&client->dev, "insufficient I2C functionality\n");
		goto exit;
	}

	adv7525 = kzalloc(sizeof(struct adv7525_dev), GFP_KERNEL);
	if (!adv7525) {
		err = -ENOMEM;
		dev_err(&client->dev, "out of memory\n");
		goto exit;
	}

	adv7525->pdata = pdata;
	adv7525->client = client;
	i2c_set_clientdata(client, adv7525);

	if (gpio_is_valid(adv7525->pdata->power_gpio)) {
		err = gpio_request(pdata->power_gpio, "adv7525");
		if (err) {
			dev_err(&client->dev, "gpio_request\n");
			goto exit_free;
		}

		err = gpio_direction_output(pdata->power_gpio, 1);
		if (err) {
			dev_err(&client->dev, "gpio_direction_output\n");
			goto exit_gpio;
		}
	}

	adv7525->hdmi_sdev.name = "hdmi";
	err = switch_dev_register(&adv7525->hdmi_sdev);
	if (unlikely(err)) {
		dev_err(&client->dev,
			"unable to register to switch framework.\n");
		goto exit_gpio;
	}

	mutex_init(&adv7525->mutex);
	INIT_WORK(&adv7525->sense_work, adv7525_sense_work);

#ifdef CONFIG_HAS_EARLYSUSPEND
	adv7525->early_suspend.suspend = adv7525_early_suspend;
	adv7525->early_suspend.resume = adv7525_late_resume;
	adv7525->early_suspend.level = EARLY_SUSPEND_LEVEL_DISABLE_FB;

	register_early_suspend(&adv7525->early_suspend);
#endif

	adv7525->attr_group = &adv7525_attr_group;
	err = sysfs_create_group(&adv7525->hdmi_sdev.dev->kobj,
				 adv7525->attr_group);
	if (err) {
		dev_err(&client->dev,
			"unable to register sysfs attributes.\n");
		goto exit_early;
	}

	err = adv7525_init_controller(adv7525);
	if (err) {
		dev_err(&client->dev, "unable to power up controller.\n");
		goto exit_sysfs;
	}

	adv7525->irq = client->irq;
	err = request_irq(adv7525->irq, adv7525_isr,
	                  IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING,
	                 client->name, adv7525);
	if (err) {
		dev_err(&client->dev, "unable to request interrupt.\n");
		goto exit_sysfs;
	}

	err = snd_soc_register_codec(&client->dev,
	                             &soc_codec_dev_adv7525_hdmi,
	                             &adv7525_hdmi_dai, 1);
	if (err) {
		dev_err(&client->dev, "codec registration failed.\n");
		goto exit_irq;
	}

	adv7525_power_control(adv7525, ADV7525_MODE_MONITOR);
	adv7525->mode = ADV7525_MODE_MONITOR;

	adv7525->dentry = debugfs_create_file("adv7525", S_IFREG | S_IRUGO,
	                                      NULL, adv7525,
	                                      &adv7525_debugfs_operations);
	if (IS_ERR(adv7525->dentry))
		adv7525->dentry = NULL;

	dev_info(&client->dev, "adv7525 registered, chip revision 0x%x\n",
	         adv7525_read_reg(adv7525, ADV7525_REG_CHIP_REVISION));

	return 0;

exit_irq:
	free_irq(adv7525->irq, adv7525);
exit_sysfs:
	sysfs_remove_group(&client->dev.kobj, adv7525->attr_group);
exit_early:
	switch_dev_unregister(&adv7525->hdmi_sdev);
	unregister_early_suspend(&adv7525->early_suspend);
exit_gpio:
	if (gpio_is_valid(adv7525->pdata->power_gpio))
		gpio_free(pdata->power_gpio);
exit_free:
	kfree(adv7525);
exit:
	return err;
}

static int __devexit
adv7525_remove(struct i2c_client *client)
{
	struct adv7525_platform_data *pdata = client->dev.platform_data;
	struct adv7525_dev *adv7525 = i2c_get_clientdata(client);

	debugfs_remove(adv7525->dentry);
	snd_soc_unregister_codec(&client->dev);
	free_irq(adv7525->irq, adv7525);
	sysfs_remove_group(&client->dev.kobj, adv7525->attr_group);
	switch_dev_unregister(&adv7525->hdmi_sdev);
	unregister_early_suspend(&adv7525->early_suspend);
	if (gpio_is_valid(adv7525->pdata->power_gpio))
		gpio_free(pdata->power_gpio);
	kfree(adv7525);

	return 0;
}

static const struct i2c_device_id adv7525_id[] = {
	{ "adv7525", 0 },
	{}
};

static struct i2c_driver adv7525_i2c_driver = {
	.driver = {
		.name	= "adv7525",
	},
	.probe    = adv7525_probe,
	.remove   = __devexit_p(adv7525_remove),
	.id_table = adv7525_id,
};

static int __init
adv7525_init(void)
{
	return i2c_add_driver(&adv7525_i2c_driver);
}

static void __exit
adv7525_exit(void)
{
	i2c_del_driver(&adv7525_i2c_driver);
}

MODULE_DEVICE_TABLE(i2c, adv7525_id);

MODULE_DESCRIPTION("ADV7525 driver");
MODULE_AUTHOR("DSP Group");
MODULE_LICENSE("GPL");

module_init(adv7525_init);
module_exit(adv7525_exit);
