/*
 * arch/arm/mach-dmw/css/netlink.c - generic cordless netlink interface to user
 *                              space
 *
 * Using netlink, a socket interface to Linux user space is implemented that
 * allows sending coma messages between user space and cordless.
 *
 * Copyright (C) 2007 NXP Semiconductors
 * Copyright (C) 2008 - 2011 DSP Group Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * An explanation about the strange use of nlh->nlmsg_seq.
 *
 * Netlink forces the application to only send buffers that are aligned to 4 bytes.
 * Netlink will round up the size of the message if it is not aligned correctly.
 * Netlink will also round up the message size when it sends messages to the application
 * from the kernel.
 *
 * This creates a problem when sending messages that don't have a length field internally
 * because the size of the message changes in the kernel <==> user-space interface.
 * To overcome this problem, the 2 most significant bits (31, 30) of the Sequence field 
 * of the Netlink Message Header (nlh->nlmsg_seq) are used to pass the value of
 * ALIGNED(length) - Length.  This is the same as (Length mod 4).
 * Later, the real message length is recreated using this information.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/socket.h>
#include <linux/list.h>
#include <linux/netlink.h>
#include <linux/workqueue.h>
#include <linux/spinlock.h>
#include <net/sock.h>

#include "coma.h"

#ifndef NETLINK_CFIFO_SIZE
#define NETLINK_CFIFO_SIZE	10000
#endif

#define NETLINK_KERNEL_MASK	0xffff0000UL


//#define COMA_DEBUG
//#define COMA_DEBUG_CONTENT

/*
 * The following definitions are expected in the file that includes this file:
 * SERVICE_NAME     - name of service, e.g., "coma-xxx"
 * SERVICE_ID       - id of service
 * NETLINK_ID         - Linux kernel netlink id (see "include/linux/netlink.h")
 * NETLINK_CFIFO_SIZE - size of cfifo (optional, default: 10000)
 */

static struct cfifo *netlink_l2c;
static struct cfifo *netlink_c2l;
static unsigned int mem_size;
static unsigned int mem_handle;
static DEFINE_MUTEX(netlink_mutex);
static struct sock *netlink_sock;
static struct workqueue_struct *tx_wq = NULL;
static struct list_head tx_work = LIST_HEAD_INIT(tx_work);
static spinlock_t lock, handler_list_lock;
static int netlink_initialized = 0;
static struct delayed_work *txw = NULL;

static int
netlink_setup(void)
{
	netlink_initialized = 1;

	return 0;
}

static void
netlink_remove(void)
{
	netlink_initialized = 0;
}

static int
coma_create_netlink_message(void *payload, unsigned int payload_size)
{
	return coma_create_message(SERVICE_ID, SERVICE_ID,
	                           NULL, 0, payload, payload_size);
}

#ifdef NETLINK_KERNEL_INTERFACE
typedef void (*t_netlink_kernel_cb)(void *, unsigned int);

struct cordless_nl_handler {
	struct list_head list;
	t_netlink_kernel_cb cb;
	unsigned long id;
};

static LIST_HEAD(handler_list);

static t_netlink_kernel_cb
netlink_find_kernel_callback(unsigned long id)
	__acquires(handler_list_lock)
	__releases(handler_list_lock)
{
	struct cordless_nl_handler *handler;

	spin_lock_bh(&handler_list_lock);
	list_for_each_entry(handler, &handler_list, list) {
		if (handler->id == id) {
			spin_unlock_bh(&handler_list_lock);
			return handler->cb;
		}
	}
	spin_unlock_bh(&handler_list_lock);

	return NULL;
}

static int
netlink_register_kernel_handler(t_netlink_kernel_cb cb, unsigned long id)
	__acquires(handler_list_lock)
	__releases(handler_list_lock)
{
	struct cordless_nl_handler *handler;

	if (netlink_find_kernel_callback(id) != NULL)
		return -1;

	handler = kzalloc(sizeof(struct cordless_nl_handler), GFP_KERNEL);
	if (!handler)
		return -ENOMEM;

	INIT_LIST_HEAD(&handler->list);
	handler->cb = cb;
	handler->id = id;

	spin_lock_bh(&handler_list_lock);
	list_add(&handler->list, &handler_list);
	spin_unlock_bh(&handler_list_lock);

	return 0;
}

static int
netlink_remove_kernel_handler(unsigned long id)
	__acquires(handler_list_lock)
	__releases(handler_list_lock)
{
	struct cordless_nl_handler *handler;

	spin_lock_bh(&handler_list_lock);
	list_for_each_entry(handler, &handler_list, list) {
		if (handler->id == id) {
			list_del(&handler->list);
			kfree(handler);
			break;
		}
	}
	spin_unlock_bh(&handler_list_lock);

	return 0;
}
#endif

static int
netlink_to_stack(unsigned char *buf, unsigned int size)
{
	int ret;

	if (!netlink_initialized)
		return -EFAULT;

	mutex_lock(&netlink_mutex);

	ret = coma_create_netlink_message(buf, size);
	if (ret == 0)
		coma_signal(SERVICE_ID);
	else
		printk(KERN_ERR "%s: cfifo/message error\n", SERVICE_NAME);

	mutex_unlock(&netlink_mutex);

	return ret;
}

struct tx_data {
	void *data;
	unsigned int size;
	struct sock *netlink_sock;
	unsigned long pid;
	struct list_head list;
};

static void tx_work_func(struct work_struct *work)
{
	struct tx_data *d;
	struct sk_buff *skb;
	struct nlmsghdr *nlh;
	void *payload;
	int ret;
	struct delayed_work *tx_w_current = container_of(work, struct delayed_work, work);
	struct delayed_work *tx_w;
	unsigned long flags;

	spin_lock_irqsave(&lock, flags);
	while (!list_empty(&tx_work)) {
		d = list_first_entry(&tx_work, struct tx_data, list);
		skb = alloc_skb(NLMSG_SPACE(d->size), GFP_ATOMIC);
		if (!skb) {
			printk(KERN_ERR "%s: alloc_skb(size = %d) returned NULL\n",
			       SERVICE_NAME, NLMSG_SPACE(d->size));
			break;
		}
		nlh = NLMSG_PUT(skb, 0, 0x123, NLMSG_DONE, d->size);
		/* data */
		payload = NLMSG_DATA(nlh);
		memcpy(payload, d->data, d->size);
		nlh->nlmsg_seq = (NLMSG_ALIGN(d->size) - d->size) << 30;
		NETLINK_CB(skb).pid = d->pid;
		NETLINK_CB(skb).dst_group = 1;

		ret = netlink_unicast(d->netlink_sock, skb, d->pid, MSG_DONTWAIT);
		if (ret == -ECONNREFUSED || ret == -EAGAIN) {
			/* again */
			tx_w = kmalloc(sizeof(*tx_w), GFP_ATOMIC);
			INIT_DELAYED_WORK(tx_w, tx_work_func);
			txw = tx_w;
			queue_delayed_work(tx_wq, tx_w, HZ / 100);
			break;
		} else {
			txw = NULL;
			kfree(d->data);
			kfree(d);
			list_del(&d->list);
		}
		continue;
nlmsg_failure:
		kfree_skb(skb);
		break;
	}
	kfree(tx_w_current);
	spin_unlock_irqrestore(&lock, flags);
}

static void
netlink_send(void *data, unsigned int size)
	__acquires(handler_list_lock)
	__releases(handler_list_lock)
	__acquires(lock)
	__releases(lock)
{
	struct sk_buff *skb;
	struct nlmsghdr *nlh;
	void *payload;
	unsigned long pid = 0;
	int ret = 0;
#ifdef COMA_DEBUG
	int j;
#endif
	struct tx_data *tx_d;
	struct delayed_work *tx_w = NULL;

	if (size < 0) {
		printk(KERN_ERR "%s: netlink_send wrong size %d\n", SERVICE_NAME, size);
		return;
	}

#ifdef COMA_DEBUG_CONTENT
    printk("KERN_INFO NETLINK SEND %s ########################################################\n", SERVICE_NAME);
    for (j = 0; j < size; j++)
    {
        if (j && (j % 16 == 0))
        {
            printk(KERN_INFO "\n");
        }
        printk("%2.2X ", ((unsigned char *)data)[j]);
    }
    printk(KERN_INFO "\n");
    printk(KERN_INFO "####################################################################\n");
#endif

#ifdef NETLINK_UNICAST
	pid = SERVICE_ID;
#endif

#ifdef NETLINK_KERNEL_INTERFACE
	if (pid & NETLINK_KERNEL_MASK) {
		struct cordless_nl_handler *handler;
		int id = pid & 0xffff;

		/* send to kernel handler, not to user space */
		if (id) {
			/* unicast */
			t_netlink_kernel_cb cb;

			cb = netlink_find_kernel_callback(id);
			if (cb)
				cb(data, size);
			return;
		}

		/* broadcast */
		spin_lock(&handler_list_lock);
		list_for_each_entry(handler, &handler_list, list) {
			if (handler->cb)
				handler->cb(data, size);
		}
		spin_unlock(&handler_list_lock);

		return;
	}
#endif /* NETLINK_KERNEL_INTERFACE */
	skb = alloc_skb(NLMSG_SPACE(size), GFP_ATOMIC);
	if (!skb) {
		printk(KERN_ERR "%s: alloc_skb(size = %d) returned NULL\n",
		       SERVICE_NAME, NLMSG_SPACE(size));
		return;
	}

	nlh = NLMSG_PUT(skb, 0, 0x123, NLMSG_DONE, size);

	/* data */
	payload = NLMSG_DATA(nlh);
	memcpy(payload, data, size);

	nlh->nlmsg_seq = (NLMSG_ALIGN(size) - size) << 30;   /// See top of file for explanation

#ifdef COMA_DEBUG
	printk("%s(size = %d): sending to pid = %lu\n", __FUNCTION__, size, pid);
#endif

	NETLINK_CB(skb).pid = pid;
	NETLINK_CB(skb).dst_group = 1;

	/* send */
	if (pid) {
		spin_lock(&lock);
		ret = netlink_unicast(netlink_sock, skb, pid, MSG_DONTWAIT);
		if (ret == -ECONNREFUSED || ret == -EAGAIN) {
			tx_d = kmalloc(sizeof(struct tx_data), GFP_ATOMIC);
			tx_d->data = kmalloc(size, GFP_ATOMIC);
			memcpy(tx_d->data, data, size);
			tx_d->size = size;
			tx_d->netlink_sock = netlink_sock;
			tx_d->pid = pid;
			if (list_empty(&tx_work)) {
				list_add_tail(&tx_d->list, &tx_work);
				tx_w = kmalloc(sizeof(*tx_w), GFP_ATOMIC);
				txw = tx_w;
				INIT_DELAYED_WORK(tx_w, tx_work_func);
				queue_delayed_work(tx_wq, tx_w, HZ / 100);
			} else {
				list_add_tail(&tx_d->list, &tx_work);
			}
		}
		spin_unlock(&lock);
	}

	return;

nlmsg_failure:
	kfree_skb(skb);
}

static int
netlink_process_message(struct cmsg *cmsg)
{
	int ret = 0;

	switch(cmsg->type) {
	case SERVICE_ID:
		netlink_send(cmsg->payload, cmsg->payload_size);
		break;
	default:
		printk(KERN_ERR "%d: Coma message is not handled\n", SERVICE_ID);
		ret = -1;
		break;
	}

	return ret;
}

static void
netlink_receive_message(struct nlmsghdr *nlh)
{
	int size = nlh->nlmsg_len - NLMSG_HDRLEN;
	char *data = NLMSG_DATA(nlh);

#ifdef COMA_DEBUG
	printk("%s: called %s\n", SERVICE_NAME, __FUNCTION__);
	printk("%s: got message with len = %d\n", SERVICE_NAME, size);
#endif

	size -= (nlh->nlmsg_seq >> 30);   /// See top of file for explanation
	netlink_to_stack(data, size);
}

static void
netlink_input(struct sk_buff *skb)
{
	struct nlmsghdr *nlh;

#ifdef COMA_DEBUG
	printk("%s: called %s()\n", SERVICE_NAME, __FUNCTION__);
#endif

	if (!netlink_initialized)
		return;

	nlh = nlmsg_hdr(skb);
	if (!NLMSG_OK(nlh, skb->len)) {
		printk(KERN_ERR "%s: received corrupt netlink message\n",
		       SERVICE_NAME);
		return;
	}

	/* process message from the application */
	netlink_receive_message(nlh);
}

static int
netlink_init(void)
{
	int ret;

	tx_wq = create_workqueue(SERVICE_NAME "_tx_wq");
	if (!tx_wq) {
		printk(KERN_ERR "%s: could not create workqueue for service\n", SERVICE_NAME);
		ret = -EFAULT;
		goto out;
	}

	spin_lock_init(&lock);
	spin_lock_init(&handler_list_lock);
	netlink_sock = netlink_kernel_create(&init_net, NETLINK_ID, 0,
	                                     netlink_input, NULL,
	                                     THIS_MODULE);
	if (!netlink_sock) {
		ret = -EFAULT;
		goto err_free_wq;
	}

	if (!netlink_l2c) {
		mem_size = cfifo_alloc(NETLINK_CFIFO_SIZE, &netlink_l2c, NETLINK_CFIFO_SIZE, &netlink_c2l, &mem_handle);
		if (mem_size < 0) {
			ret = -CFIFO_NOMEM;
			goto err_free_wq;
		}
	} else {
		cfifo_reset(netlink_l2c);
		cfifo_reset(netlink_c2l);
	}

	ret = coma_register(SERVICE_ID, SERVICE_NAME,
	                    netlink_l2c, netlink_c2l,
	                    netlink_process_message, netlink_setup,
	                    netlink_remove);
	if (ret < 0) {
		ret = -EFAULT;
		goto err_free_cfifo;
	}

	printk(KERN_INFO "%s(%d): netlink interface registered\n", SERVICE_NAME, SERVICE_ID);

	return 0;

err_free_cfifo:
	cfifo_free((char *)netlink_l2c, mem_size, mem_handle);
	netlink_l2c = 0;
	netlink_c2l = 0;
err_free_wq:
	destroy_workqueue(tx_wq);
out:
	return ret;
}

static void
netlink_exit(void)
{
	netlink_initialized = 0;
	coma_deregister(SERVICE_ID);
	spin_lock(&lock);
	sock_release(netlink_sock->sk_socket);
	if (txw != NULL)
		cancel_delayed_work_sync(txw);
	spin_unlock(&lock);
	destroy_workqueue(tx_wq);
//	cfifo_free((char *)netlink_l2c, mem_size, mem_handle);
}
