/*
 * arch/arm/mach-dmw/css/csignaling.c - signaling between domains
 *
 * This module provides an infrastructure for signaling new messages to the
 * other domain and for informing the corresponding service of a new message
 * sent by the other domain.
 * Signaling is either implemented through software-triggered interrupts if
 * both domains are hosted on the same CPU (VegaFB) or through the "semaphore
 * exchange center" if each domain runs on a dedicated processer (DW).
 *
 * Copyright (C) 2009 DSP Group Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/err.h>
#include <asm/io.h>
#include <mach/platform.h>
#include <mach/sec.h>
#include "coma.h"
#include "coma-service.h"
#include "cfifo.h"
#include "ccommon.h"

/*
 * External interface:
 * coma_signaling_init()  - set up signaling infrastructure
 * coma_signal(id)        - signal a new outgoing message for a specific
 *                          service to other domain
 *
 * Functions to be provided by the cordless manager:
 * coma_receive(id)       - signal a new incoming message from the other
 *                          domain for a specific service
 */

extern void coma_receive(int id);

#ifndef CONFIG_CSS_UART

static struct sec_msg *sec[COMA_NUM_SERVICES];

#endif /* CONFIG_CSS_UART */

void
coma_signal(int id)
{
#ifdef CONFIG_CSS_UART

	unsigned long flags;

	local_save_flags(flags);
	local_irq_disable();
	local_fiq_disable();
	//writel((1<<31), IO_ADDRESS(DW_INTC_BASE) + DW_INTC_FSTATUS);
	local_irq_restore(flags);

#else /* CONFIG_CSS_UART */

	sec_msg_trigger(sec[id], 0xdeadbeef);

#endif /* CONFIG_CSS_UART */
}
EXPORT_SYMBOL(coma_signal);

/*
 * interrupt handling
 */
static void
coma_process_irq(unsigned long dummy)
{
	int i;

	for (i = 0; i < COMA_NUM_SERVICES; i++)
		coma_receive(i);
}

DECLARE_TASKLET(coma_tasklet, coma_process_irq, 0);

#ifdef CONFIG_CSS_UART

static irqreturn_t
coma_irq_handler(int irq, void *dev_id)
{
	/* clear sw int done in irq.c */

	tasklet_schedule(&coma_tasklet);
	return IRQ_HANDLED;
}

#else /* CONFIG_CSS_UART */

static void sec_handler(int id, unsigned long data, void *context)
{
	/* clear sw int done in irq.c */
	tasklet_schedule(&coma_tasklet);
}

#endif /* CONFIG_CSS_UART */

int coma_signaling_init(void)
{

#ifdef CONFIG_CSS_UART

	int ret;
	/* use soft IRQ */

	ret = request_irq(DMW_IID_SW4, coma_irq_handler, 0, "coma", NULL);
	if (ret != 0) {
		printk(KERN_ERR "coma: cannot register irq.\n");
		return ret;
	}

#else /* CONFIG_CSS_UART */

	int i;
	/* use SEC mechanism */
	BUG_ON((SEC_MSG_COMA_START + COMA_NUM_SERVICES) > SEC_MSG_COMA_END);
	for (i = 0; i < COMA_NUM_SERVICES; i++) {
		sec[i] = sec_msg_register(sec_handler, i + SEC_MSG_COMA_START,
					  0, NULL);
		if (IS_ERR_OR_NULL(sec[i])) {
			printk(KERN_ERR "COMA: cannot register SEC %d\n", i);
			return -EBUSY;
		}
	}

	printk(KERN_DEBUG "COMA: sec init\n");

#endif /* CONFIG_CSS_UART */

	return 0;
}
EXPORT_SYMBOL(coma_signaling_init);

void coma_signaling_exit(void)
{

#ifdef CONFIG_CSS_UART

	free_irq(DMW_IID_SW4, NULL);

#else /* CONFIG_CSS_UART */

	int i;
	for (i = 0; i < COMA_NUM_SERVICES; i++) {
		sec_msg_deregister(sec[i]);
	}
#endif /* CONFIG_CSS_UART */

}
EXPORT_SYMBOL(coma_signaling_exit);
