/*
 * gpio_vibrator.c - GPIO vibrator driver
 *
 * (C) Copyright 2011, DSP Group Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/err.h>
#include <linux/hrtimer.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/regulator/consumer.h>
#include <linux/gpio.h>
#include <linux/gpio_vibrator.h>

/* TODO: replace with correct header */
#include "../staging/android/timed_output.h"

MODULE_AUTHOR("DSP Group Inc.");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Vibrator driver for motors connected via GPIO (on/off)");

struct gpio_vibrator {
	struct gpio_vibrator_pdata *pdata;
	struct regulator *vcc;
	struct work_struct work;
	struct hrtimer timer;
	spinlock_t lock;
	int enabled;

	struct timed_output_dev timed_output;
};

static void gpio_vibrator_update(struct work_struct *work)
{
	struct gpio_vibrator *vib =
		container_of(work, struct gpio_vibrator, work);
	int enabled = hrtimer_active(&vib->timer);

	gpio_set_value(vib->pdata->onoff, enabled);

	if (!vib->enabled && enabled) {
		regulator_enable(vib->vcc);
		vib->enabled = 1;
	} else if (vib->enabled && !enabled) {
		regulator_disable(vib->vcc);
		vib->enabled = 0;
	}
}

static void gpio_vibrator_enable(struct timed_output_dev *dev, int value)
{
	struct gpio_vibrator *vib =
		container_of(dev, struct gpio_vibrator, timed_output);
	unsigned long flags;

	spin_lock_irqsave(&vib->lock, flags);

	hrtimer_cancel(&vib->timer);
	if (value > 0) {
		value = (value > 15000 ? 15000 : value);
		hrtimer_start(&vib->timer,
			ktime_set(value / 1000, (value % 1000) * 1000000),
			HRTIMER_MODE_REL);
	}

	spin_unlock_irqrestore(&vib->lock, flags);

	schedule_work(&vib->work);
}

static int gpio_vibrator_get_time(struct timed_output_dev *dev)
{
	struct gpio_vibrator *vib =
		container_of(dev, struct gpio_vibrator, timed_output);

	if (hrtimer_active(&vib->timer)) {
		ktime_t r = hrtimer_get_remaining(&vib->timer);
		struct timeval t = ktime_to_timeval(r);
		return t.tv_sec * 1000 + t.tv_usec / 1000;
	} else
		return 0;
}

static enum hrtimer_restart dspg_vibrator_timer_func(struct hrtimer *timer)
{
	struct gpio_vibrator *vib =
		container_of(timer, struct gpio_vibrator, timer);

	schedule_work(&vib->work);
	return HRTIMER_NORESTART;
}

#if CONFIG_PM
static int gpio_vibrator_suspend(struct device *dev)
{
	struct gpio_vibrator *vib = dev_get_drvdata(dev);

	if (vib->enabled) {
		gpio_vibrator_enable(&vib->timed_output, 0);
		flush_work_sync(&vib->work);
	}

	return 0;
}

static int gpio_vibrator_resume(struct device *dev)
{
	return 0;
}

static const struct dev_pm_ops gpio_vibrator_pm_ops = {
	     .suspend	= gpio_vibrator_suspend,
	     .resume	= gpio_vibrator_resume,
};
#endif

static int __devinit gpio_vibrator_probe(struct platform_device *pdev)
{
	struct gpio_vibrator_pdata *pdata = pdev->dev.platform_data;
	struct gpio_vibrator *vib;
	int ret = 0;

	vib = kzalloc(sizeof(*vib), GFP_KERNEL);
	if (!vib)
		return -ENOMEM;

	platform_set_drvdata(pdev, vib);

	vib->pdata = pdata;
	vib->vcc = regulator_get(&pdev->dev, "Vcc");
	if (IS_ERR(vib->vcc)) {
		dev_err(&pdev->dev, "cannot get regulator\n");
		ret = PTR_ERR(vib->vcc);
		goto err_free;
	}

	INIT_WORK(&vib->work, gpio_vibrator_update);
	spin_lock_init(&vib->lock);

	hrtimer_init(&vib->timer, CLOCK_MONOTONIC, HRTIMER_MODE_REL);
	vib->timer.function = dspg_vibrator_timer_func;

	ret = gpio_request(vib->pdata->onoff, "gpio-vibrator");
	if (ret) {
		dev_err(&pdev->dev, "cannot get GPIO\n");
		goto err_put_reg;
	}

	gpio_direction_output(vib->pdata->onoff, 0);
	vib->enabled = 0;

	vib->timed_output.name = "vibrator";
	vib->timed_output.get_time = gpio_vibrator_get_time;
	vib->timed_output.enable = gpio_vibrator_enable;
	timed_output_dev_register(&vib->timed_output);

	dev_info(&pdev->dev, "GPIO vibrator initialized\n");
	return 0;

err_put_reg:
	regulator_put(vib->vcc);
err_free:
	kfree(vib);
	return ret;
}

static int __devexit gpio_vibrator_remove(struct platform_device *pdev)
{
	struct gpio_vibrator *vib = platform_get_drvdata(pdev);

	timed_output_dev_unregister(&vib->timed_output);
	gpio_free(vib->pdata->onoff);
	regulator_put(vib->vcc);
	kfree(vib);

	return 0;
}

static struct platform_driver gpio_vibrator_driver = {
	.probe		= gpio_vibrator_probe,
	.remove		= __devexit_p(gpio_vibrator_remove),
	.driver		= {
		.name	= "gpio-vibrator",
		.owner	= THIS_MODULE,
#ifdef CONFIG_PM
		.pm	= &gpio_vibrator_pm_ops,
#endif
	}
};

static int __init gpio_vibrator_init(void)
{
	return platform_driver_register(&gpio_vibrator_driver);
}

static void __exit gpio_vibrator_exit(void)
{
	platform_driver_unregister(&gpio_vibrator_driver);
}

module_init(gpio_vibrator_init);
module_exit(gpio_vibrator_exit);

