/*
 *  linux/arch/arm/mach-dmw/include/mach/sec.h
 *
 *  Copyright (C) 2010 DSP Group
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */
#ifndef SEC_H
#define SEC_H

struct sec_msg;
struct sec_sem;

#define SEC_OVERWRITE	0x2
#define SEC_DONT_CLEAR	0x4
#define SEC_FIQ		0x8

/* ==============
 *    Messages
 * ==============
 *
 * The concept of message used in this driver represents two messages in the
 * SEC memory - one in the CPU2COM and the corresponding message in the
 * COM2CPU memory. The function "sec_msg_trigger" writes to the CPU2COM
 * message, while the "sec_callbac" reads from the COM2CPU memory.
 */

/* The function sec_callback is called from FIQ or IRQ context when the CSS
 * writes to the SEC memory. Code executed from this function must not sleep.
 */
typedef void (*sec_callback)(int id, unsigned long data, void *context);

/* Write data to the semaphore memory and trigger an interrupt at the CSS. */
int  sec_msg_trigger(struct sec_msg *sem, unsigned long data);

/* Clear the status bit (needed if SEC_DONT_CLEAR is set). */
void sec_msg_clear(struct sec_msg *sem);

/*
 * Read current CSS value. Usefull if SEC_OVERWRITE was set and no callback was
 * specified.
 */
unsigned long sec_msg_read(struct sec_msg *msg);

/* Register a callback <cb> for semaphore <id> and un-mask its interrupt.
 * Prerequisite for "sec_msg_trigger". <cb> can be set to NULL if interrupts
 * from the CSS should be ignored.
 * The following flags can be specified:
 * SEC_OVERWRITE:  allow further writes to a message even if the status bit
 *                 was not cleared by the CSS
 * SEC_DONT_CLEAR: do not clear the status bit automatically when a message is
 *                 received from the CSS. If set, the function "sec_msg_clear"
 *                 can be used to clear the status bit.
 * SEC_FIQ:        call the handler directly from FIQ context
 * Returns a handle for the message, or a valid IS_ERR() condition containing
 * errno. */
struct sec_msg *sec_msg_register(sec_callback cb, int id, int flags, void *context);

/* Free the message and mask its interrupt. */
void sec_msg_deregister(struct sec_msg *sem);

#endif /* SEC_H */
