/*
 *  linux/arch/arm/mach-dmw/camera.h
 *
 *  Copyright (C) 2012 DSP Group
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 * DMW96 camera enumerator. used to associate front/back camera ports,
 * which share the same CIU, with individual camera sensors
 *
 */

#ifndef __MACH_CAMERA_H
#define __MACH_CAMERA_H

#include <media/dmw96ciu_w.h>
#include <linux/i2c.h>
#include <linux/platform_device.h>

/*-------------------
 - sensor descriptor
 -------------------*/

struct dmw_camera_port_s;
struct dmw_camera_sensor_desc_s;

/* sensor connectivity mode */
typedef enum dmw_camera_sensor_mode_e
{
	DMW_CAMERA_SENSOR_MODE_PARALLEL,
	DMW_CAMERA_SENSOR_MODE_MIPI_1LANE,
	DMW_CAMERA_SENSOR_MODE_MIPI_2LANES,
} dmw_camera_sensor_mode_t;

/* probe */
typedef int ( * dmw_camera_sensor_probe_f )( struct dmw_camera_sensor_desc_s * sensor_desc,
											 struct dmw_camera_port_s * 	   cam_port );

/* camera sensor descriptor */
typedef struct dmw_camera_sensor_desc_s
{
	/* sensor name */
	const char *			  name;

	/* I2C address */
	int						  i2c_addr;

	/* probing function, called at port initialization to detect a known sensor */
	dmw_camera_sensor_probe_f probe;

	/* connectivity mode */
	dmw_camera_sensor_mode_t  mode;

} dmw_camera_sensor_desc_t;

/*----------------------
 - sensor platform data
 ----------------------*/

/* NOTE: optional, can either be supplied by the board (for providing private data
 * to the sensor through sensor_pdata) or left empty, in which case the sensor driver
 * should use a default one */
typedef struct dmw_camera_sensor_platform_data_s
{
	/* sensor descriptor, set in runtime by the probe() function of the sensor
	 * platform device, shouldn't be touched by the user */
	void * __sensor_desc;

	/* sensor-specific data, passed by the board to the sensor driver */
	void * sensor_pdata;

} dmw_camera_sensor_platform_data_t;

/*-------------------
 - port data
 -------------------*/

typedef enum dmw_camera_port_type_e
{
	DMW_CAMERA_PORT_PARALLEL,
	DMW_CAMERA_PORT_MIPI,
} dmw_camera_port_type_t;

/* port power control */
typedef int ( * dmw_camera_port_power_ctl_f )( struct dmw_camera_port_s * port,
											   enum v4l2_power 			  power,
											   dmw_camera_sensor_desc_t * sensor );

/* port clocking control */
typedef int ( * dmw_camera_port_clock_ctl_f )( struct dmw_camera_port_s * port,
											   unsigned long 			  xclkfreq );

/* camera port (front, back, etc) */
typedef struct dmw_camera_port_s
{
	/*----------------------
	 - configuration
	 ----------------------*/

	/* port name */
	const char *				name;

	/* port pdata, for use by the board implementation */
	void * 				   		pdata;

	dmw_camera_port_power_ctl_f power_ctl;
	dmw_camera_port_clock_ctl_f clock_ctl;

	/* list of sensors that might be found on this port (only one can be successfuly probed) */
	struct platform_device **   sensor_list;

	/* number of sensors in list */
	unsigned long				num_sensors;

	/* i2c adapter number to which this port is connected */
	int							i2c_adapter_num;

	/* CIU hardware data. returned as the port private data to V4L */
	struct dmw96cam_hw_config   hw_config;

	/*----------------------
	 - set at runtime
	 ----------------------*/

	/* currently used sensor */
	dmw_camera_sensor_desc_t * sensor;

	/* i2c client representing the sensor connected to the port. created at runtime */
	struct i2c_client *		   i2c_client;

	/* board info for i2c registration */
	struct i2c_board_info 	   i2c_board_info;

	/* i2c reference counter */
	int						   i2c_ref;

} dmw_camera_port_t;

/*-------------------
 - camera data
 -------------------*/

// TODO REMOVE?
/* camera subsystem */
typedef struct dmw_camera_s
{
	/* front/back ports */
	dmw_camera_port_t front;
	dmw_camera_port_t back;
} dmw_camera_t;

/************************
 * interfaces
 ************************/

/* port power control, called by sensors */
int 				dmw_camera_port_power_ctl( dmw_camera_port_t * port,
							   	   	   	   	   enum v4l2_power	   power );

/* port clocking control, called by sensors */
int 				dmw_camera_port_clock_ctl( dmw_camera_port_t * port,
							   	   	   	   	   unsigned long 	   xclkfreq );

/* start using i2c */
struct i2c_client * dmw_camera_get_i2c( dmw_camera_port_t * 	   port,
										dmw_camera_sensor_desc_t * sensor_desc );

/* stop using i2c */
void 				dmw_camera_put_i2c( dmw_camera_port_t * port );

/* associate sensor with port. called after sensor is up */
void 				dmw_camera_associate_sensor_with_port( dmw_camera_port_t * 	   	  port,
														   dmw_camera_sensor_desc_t * sensor_desc );

/* initialize port */
int 				dmw_camera_initialize_port( dmw_camera_port_t * port );

#endif
