/*
 * arch/arm/mach-dmw/css/coma-dua.c - cordless DUA / netlink interface to user space
 *
 * Using netlink, a socket interface to Linux user space is implemented that
 * allows sending DUA messages between user space and cordless.
 *
 * Copyright (C) 2007 NXP Semiconductors
 * Copyright (C) 2008 - 2011 DSP Group Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include "coma-service.h"

#define SERVICE_NAME		"coma-dua"
#define SERVICE_ID		COMA_SERVICE_DUA
#define NETLINK_ID		NETLINK_DUA
#define NETLINK_CFIFO_SIZE	10240

#define NETLINK_UNICAST
#define NETLINK_KERNEL_INTERFACE

#include "netlink.c"

typedef void (*t_dua_kernel_cb) (void *, unsigned int);

int
dua_pack_message(int caller_id, int transaction_id, int msg_id,
		 const long *params, int nr_params,
		 const void *data, int data_size, char *buf, int max_size)
{
	int size = 0;
	int i;

	/* caller_id (0 means broadcast) */
	if (size + sizeof(int) > max_size)
		return -1;
	memcpy(buf, (void *)&(caller_id), sizeof(int));
	buf = (char *)((int *)buf + 1);
	size += sizeof(int);

	/* transaction_id */
	if (size + sizeof(int) > max_size)
		return -1;
	memcpy(buf, (void *)&(transaction_id), sizeof(int));
	buf = (char *)((int *)buf + 1);
	size += sizeof(int);

	/* msg_id */
	if (size + sizeof(int) > max_size)
		return -1;
	memcpy(buf, (void *)&(msg_id), sizeof(int));
	buf = (char *)((int *)buf + 1);
	size += sizeof(int);

	/* number of parameters */
	if (size + sizeof(int) > max_size)
		return -1;
	memcpy(buf, (void *)&(nr_params), sizeof(int));
	buf = (char *)((int *)buf + 1);
	size += sizeof(int);

	/* parameters */
	for (i = 0; i < nr_params; i++) {
		if (size + sizeof(int) > max_size)
			return -1;
		memcpy(buf, (void *)&(params[i]), sizeof(int));
		buf = (char *)((int *)buf + 1);
		size += sizeof(int);
	}

	/* size of data block */
	if (size + sizeof(int) > max_size)
		return -1;
	memcpy(buf, (void *)&(data_size), sizeof(int));
	buf = (char *)((int *)buf + 1);
	size += sizeof(int);

	data_size = ((data_size + 3) / 4) * 4;

	/* data */
	if (size + data_size > max_size)
		return -1;
	memcpy(buf, data, data_size);
	buf += data_size;
	size += data_size;

	return size;
}

EXPORT_SYMBOL(dua_pack_message);

int
dua_unpack_message(int *caller_id, int *transaction_id, int *msg_id,
		   long *params, int max_params,
		   void *data, int *data_size, char *buf, int max_size)
{
	int size = 0;
	int i;
	int nr_params;

	if (size + sizeof(int) > max_size)
		return -1;
	*caller_id = *(int *)((char *)buf + size);
	size += sizeof(int);

	if (size + sizeof(int) > max_size)
		return -1;
	*transaction_id = *(int *)((char *)buf + size);
	size += sizeof(int);

	if (size + sizeof(int) > max_size)
		return -1;
	*msg_id = *(int *)((char *)buf + size);
	size += sizeof(int);

	if (size + sizeof(int) > max_size)
		return -1;
	nr_params = *(int *)((char *)buf + size);
	size += sizeof(int);

	if (nr_params > max_params)
		return -1;

	/* unpack parameters
	 */
	for (i = 0; i < nr_params; i++) {
		if (size + sizeof(int) > max_size)
			return -1;
		params[i] = *(int *)((char *)buf + size);
		size += sizeof(int);
	}

	/* unpack data */
	if (size + sizeof(int) > max_size)
		return -1;
	*data_size = *(int *)((char *)buf + size);
	size += sizeof(int);

	if (size + *data_size > max_size)
		return -1;
	memcpy(data, (char *)buf + size, *data_size);
	size += *data_size;

	size = ((size + 3) / 4) * 4;

	return size;
}

EXPORT_SYMBOL(dua_unpack_message);

int dua_register_kernel_handler(t_dua_kernel_cb cb, unsigned long id)
{
	return netlink_register_kernel_handler((t_netlink_kernel_cb) cb, id);
}

EXPORT_SYMBOL(dua_register_kernel_handler);

int dua_remove_kernel_handler(unsigned long id)
{
	return netlink_remove_kernel_handler(id);
}

EXPORT_SYMBOL(dua_remove_kernel_handler);

int dua_kernel_to_cordless(unsigned char *data, unsigned int size)
{
	return netlink_to_stack(data, size);
}

EXPORT_SYMBOL(dua_kernel_to_cordless);

int dua_kernel_to_linux(unsigned char *data, unsigned int size)
{
	netlink_send(data, size);

	return 0;
}

EXPORT_SYMBOL(dua_kernel_to_linux);

void dua_service_init(void)
{
	netlink_init();
}

void dua_service_exit(void)
{
	netlink_exit();
}
