/*
 * arch/arm/mach-dmw/css/cfifo.c - cordless FIFOs
 *
 * Cordless FIFOs store variable-length messages in a ring buffer. This FIFO
 * implementation is safe for parallel accesses of not more than one reader
 * and one writer.
 * cordless_fifo_msg_put() copies a message to the internal buffer of the
 * FIFO, while cordless_fifo_msg_get() retrieves the pointer to the next
 * message (if any) and its length. It does not copy the data, i.e. the caller
 * does not need to allocate memory in order to access the message. The FIFO
 * has to be informed using cordless_fifo_msg_processed() if the message is
 * processed and can be overwritten by the FIFO. A message is stored in a
 * consecutive memory area, i.e., it is not wrapped around at the end of the
 * FIFO as in many other ring buffer implementations.
 *
 * Copyright (C) 2007 NXP Semiconductors
 * Copyright (C) 2008 - 2011 DSP Group Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/err.h>
#include <linux/spinlock.h>
#include <linux/vmalloc.h>
#include <linux/slab.h>
#include <linux/dma-mapping.h>
#include "cfifo.h"


#ifdef CONFIG_DEBUG_FS
static LIST_HEAD(cfifo_list);

void * get_cfifo_list_head(void){
	return &cfifo_list;
}

#endif

/**
 * cfifo_init - allocates a new FIFO using a preallocated buffer
 * @buffer: the preallocated buffer to be used.
 * @size: the size of the internal buffer.
 *
 */
static struct cfifo *
cfifo_init(unsigned char *buffer, unsigned int size,  unsigned int self_phys)
{
	struct cfifo *cfifo;

	cfifo = (struct cfifo *)buffer;

	size = CFIFO_PAD(size);

	cfifo->buffer = buffer + CFIFO_PAD(sizeof(*cfifo));
	cfifo->size = size;
	cfifo->self_phys = self_phys;

	spin_lock_init(&cfifo->spinlock);

	cfifo_reset(cfifo);

	return cfifo;
}

/*
 * cfifo_free - frees the cfifo
 * @cfifo: the fifo to be freed.
 */
void cfifo_free(char *buffer, unsigned int size, unsigned int handle)
{
#ifdef CONFIG_DEBUG_FS
cfifo_node_t *pnode;
#endif
#ifdef CONFIG_CSS_UART

	vfree(buffer);

#else /* CONFIG_CSS_UART */

	dma_free_coherent(NULL, size, buffer, handle);
#ifdef CONFIG_DEBUG_FS
	list_for_each_entry(pnode,&cfifo_list,list){
		if((unsigned int)buffer == (unsigned int)pnode->pbuf){
			list_del(&pnode->list);
			kfree(pnode);
			break;
		}
	}
#endif

 #endif /* CONFIG_CSS_UART */
}
EXPORT_SYMBOL(cfifo_free);


/*
 * cfifo_alloc - allocates a new cfifos staff and its internal buffer
 * @size1 nad @size2: the sizes of the internal buffers to be allocated.
 */
int
cfifo_alloc(unsigned int size1, struct cfifo **fifo1, unsigned int size2, struct cfifo **fifo2, unsigned int *handle)
{
	unsigned char *buffer;
	dma_addr_t dma_handle;
	int size;
#ifdef CONFIG_DEBUG_FS	
	cfifo_node_t *p_cfifo_node=NULL;
#endif

	size = CFIFO_PAD(size1) + CFIFO_PAD(size2) + CFIFO_PAD(sizeof(struct cfifo) * 2);

#ifdef CONFIG_CSS_UART

	buffer = vmalloc(size);
	if (!buffer)
		return -CFIFO_NOMEM;

	dma_handle = 0;

#else /* CONFIG_CSS_UART */

	buffer = dma_alloc_coherent(NULL, size, &dma_handle, GFP_KERNEL);
	if (!buffer)
		return -CFIFO_NOMEM;
	
#ifdef CONFIG_DEBUG_FS	
	p_cfifo_node = (cfifo_node_t*)kmalloc(sizeof(cfifo_node_t),GFP_KERNEL);
	if(!p_cfifo_node)
		return -CFIFO_NOMEM;
	
	p_cfifo_node->pbuf = buffer;
	p_cfifo_node->size = size;

	list_add_tail(&p_cfifo_node->list, &cfifo_list);
#endif	

 #endif /* CONFIG_CSS_UART */

	printk(KERN_INFO "COMA: cfifo allocate addr 0x%X phys 0x%X\n", (int)buffer, dma_handle);

	*fifo1 = cfifo_init(buffer, size1, dma_handle);
	*fifo2 = cfifo_init(buffer + CFIFO_PAD(size1) + CFIFO_PAD(sizeof(struct cfifo)), 
						size2, dma_handle + CFIFO_PAD(size1) + CFIFO_PAD(sizeof(struct cfifo)));

	*handle = dma_handle;

	return size;
}
EXPORT_SYMBOL(cfifo_alloc);
