#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#define TASK_COMM_LEN 16

struct sched_stat {
	unsigned int start_sec, start_nsec;
	unsigned int end_sec, end_nsec;
	char name[TASK_COMM_LEN];
	int pid;
};
static struct sched_stat *stats;

struct linux_task {
	char *name;
	int pid;
};
static struct linux_task *tasks;

static int compare_tasks(const void *a, const void *b)
{
	return (((struct linux_task *)a)->pid - ((struct linux_task *)b)->pid);
}

int main(int argc, char **argv)
{
	int i, j, k, fd, first = 0, nr_stats, nr_tasks = 0, filesize;
	struct sched_stat *stat;
	unsigned int first_sec = 0, first_nsec = 0;

	if (argc != 2) {
		fprintf(stderr, "Usage: %s <in-file>\n", argv[0]);
		exit(1);
	}

	fd = open(argv[1], O_RDONLY);
	if (fd < 0) {
		fprintf(stderr, "%s: cannot open file %s\n", argv[0], argv[1]);
		exit(1);
	}

	filesize = lseek(fd, 0, SEEK_END);
	lseek(fd, 0, SEEK_SET);

	nr_stats = filesize/sizeof(struct sched_stat);

	stats = malloc(nr_stats * sizeof(struct sched_stat));
	if (!stats) {
		fprintf(stderr, "malloc failed\n");
		exit(1);
	}
	tasks = malloc(nr_stats * sizeof(struct linux_task));
	if (!tasks) {
		fprintf(stderr, "malloc failed\n");
		exit(1);
	}

	for (i = 0; i < nr_stats; i++) {
		tasks[i].name = 0;
		tasks[i].pid = 0;
	}

	for (i = 0; i < nr_stats; i++) {
		if (read(fd, &stats[i], sizeof(struct sched_stat)) != sizeof(struct sched_stat))
			break;
	}

	/* Find first entry in ringbuffer */
	for (i = nr_stats-1; i >= 0; i--) {
		if (!stats[i].name[0]) {
			first = i+1;
			break;
		}
	}
	if (first >= nr_stats)
		first = 0;

	first_sec = stats[i].start_sec;
	first_nsec = stats[i].start_nsec;

	printf("SPEED 1\nTIME 1\n");

	/* Find processes */
	for (i = 0; i < nr_stats; i++) {
		j = (i + first) % nr_stats;
		stat = &stats[j];
		if (!stat->name[0])
			continue;
		for (k = 0; k < nr_stats; k++) {
			if (tasks[k].name && (tasks[k].pid == stat->pid))
				break;
			if (!tasks[k].name) {
				tasks[k].name = stat->name;
				tasks[k].pid = stat->pid;
				nr_tasks++;
				break;
			}
		}
	}

	fprintf(stderr, "Found %d events and %d processes.\n", nr_stats, nr_tasks);

	/* Replace whitespace in process names with '_' (limitation of TimeDoctor) */
	for (i = 0; i < nr_tasks; i++) {
		for (j = 0; j < TASK_COMM_LEN; j++) {
			if (tasks[i].name[j] == ' ')
				tasks[i].name[j] = '_';
		}
	}

	/* Sort according to pid */
	qsort(tasks, nr_tasks, sizeof(struct linux_task), compare_tasks);
	for (i = 0; i < nr_tasks; i++) {
		printf("CRE 0 %d %u.%06u\n", tasks[i].pid, 0, first_nsec/1000);
		printf("NAM 0 %d %s-%d\n", tasks[i].pid, tasks[i].name, tasks[i].pid);
	}

	/* Print tasks */
	for (i = 0; i < nr_stats; i++) {
		j = (i + first) % nr_stats;
		stat = &stats[j];
		if (!stat->name[0])
			continue;
		printf("STA 0 %d %u.%06u\n", stat->pid, stat->start_sec-first_sec, stat->start_nsec/1000);
		printf("STO 0 %d %u.%06u\n", stat->pid, stat->end_sec-first_sec, stat->end_nsec/1000);
	}

	printf("END\n");

	return 0;
}
