#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2003-2009 Free Software Foundation
#
# FILE:
# PrinterDefinition.py
#
# DESCRIPTION:
# Class that loads a Postscript Printer Description (PPD) file
# Supports version 4.3 of the PPD spec (at least, we hope)
#
# NOTES:
# Spec: http://partners.adobe.com/asn/developer/pdfs/tn/5003.PPD_Spec_v4.3.pdf

__all__ = ['PrinterDefinition','InvalidPPDFormat']

import string, re
from gnue.common.utils.FileUtils import openResource
from gnue.reports.adapters.filters.Universal.Base.Adapter import UniversalError

# Just what its name implies...
class InvalidPPDFormat(UniversalError):
  pass

#
# PrinterDefinition
#
class PrinterDefinition:

  #
  # Public methods
  #
  def getCanvasSize(self):
    # TODO: real numbers
    w = 612
    h = 792

    if self.landscape:
      return (h,w)
    else:
      return (w,h)

  def getPageSize(self):
    w = 612
    h = 792
    return (w,h)

  def getBoundingBox(self):
    w = 612
    h = 792
    return (0,0,w,h)

  def getDscComments(self):
    rv = "%%%%Orientation: %s" % (self.landscape and "Landscape" or "Portrait")
    rv += "%%%%BoundingBox: %s %s %s %s" %  self.getBoundingBox()

  def getDscSetup(self):
    rv = ""
    if self.copies > 1:
      rv += "<< /NumCopies %d >> setpagedevice\n" % self.copies
    if self.landscape:
      rv += "" ### TODO
    return rv

  def getDscPageSetup(self):
    # Cache the page setup code as it (currently?)
    #  doesn't change from one page to the next.
    try:
      return self.__pageSetupCode
    except:
      rv = ""
      if self.landscape:
        rv += "90 rotate 0 -%s translate\n" % self.getPageSize()[0]
        
      self.__pageSetupCode = rv
      return rv


  #
  # Internal methods
  #
  def __init__(self, location):
    self.__symbols__ = {}
    self.__uioptions__ = {}
    self.__properties__ = {}
    self.__defaults__ = {}
    self.__values__ = {}

    self.copies = 0
    self.landscape = 0

    self.loadFile(location)

  def __getattr__(self, attr):
    if self.__dict__.has_key(attr):
      return self.__dict__[attr]
    else:
      return self.__properties__[attr]


  def __getitem__(self, attr):
    try:
      return self.__values__[attr]
    except KeyError:
      try:
        return self.__defaults__[attr]
      except:
        return ""

  def getNameAndCode(self, attr):
    return (self[attr], self.getCode(attr))

  def getCode(self, attr):
    value = self.__properties__[self[attr]]
    code = self.__properties__[attr][value][0]
    rv = "%%%%BeginFeature: %s *%s\n" % (attr, value)
    rv += code
    rv += "\n%%EndFeature\n"
    return rv

  #
  # Load/import a new file
  # Can be passed a file buffer or a file name.
  #
  def loadFile(self, location):
    # Get a file handle.
    # location can be either a string, or a buffer
    if hasattr(location,'read'):
      handle = location
      close = 0
    else:
      handle = openResource(location)
      close = 1


    #
    # Process each line of the file
    #
    line = handle.readline()
    while line != '':

      line = line.strip()

      # Skip blank lines and comments (*%).
      if line[:2] in ('','*%'):
        line = handle.readline()
        continue

      ##print "Line: %s" % line

      # Get stuff to the left of the colon
      # (Can either be a Keyword, or a Keyword + Option)
      try:
        keyword, data = line.split(':',1)
      except ValueError:
        keyword = line
        data = ''

      # Dump the leading asterick (not sure
      # if the .strip() is necessary)
      keyword = keyword[1:].strip()

      print "  Found Key Word: %s" % keyword

      # Clean up the data line
      data = data.strip()

      # Do we have an Option, or just a Keyword?
      try:
        d = keyword.split(' ')
        keyword = d[0]
        option = string.join(d[1:],' ')

        # Options can be in the format Name/Friendly Name
        # The Friendly Name part is called a "translation"
        try:
          option, option_translation = option.split('/',1)
      ##    print "  Option/Trans=[%s] [%s]/[%s]" % (keyword, option, option_translation)
        except ValueError:
          option_translation = None
      ##    print "  Option=%s" % (option)

      except ValueError:
        option = None
        option_translation = None

      # ^Data means Data is a Symbol
      if data[:1] == '^':
        # Create a wrapper, so the app developer doesn't
        # have to distinguish between data being a symbol
        # or a string.  _SymbolData acts like a string.
        data = _SymbolData(self, data[1:])

      # Double Quotes means data is Quoted... treat specially
      elif data[:1] == '"':

        # If quote ends on this line, this is a single-line quote
        if data[1:].find('"') >= 0:
          # For now, we ignore any option translation strings
          pos = data[1:].find('"')+2
          data = data[1:pos]

        # otherwise, we need to grab the rest of the data
        else:

          # Get rid of leading quote, and restore
          # the newline dropped earlier
          data = data[1:] + '\n'

          # Find the next '*End' marker
          line = handle.readline()
          while line != None and line.strip() != '*End':
            data += line
            line = handle.readline()

          # If we never hit an *End, then this isn't a valid file.
          if line == '':
            raise InvalidPPDFormat, 'Not a valid PPD file... missing *End'

          # Get rid of the closing quote
          data = data.strip()[:-1]

          # If this was not a Keyword/Option pair, then the PPD spec
          # allows for the string to contain hex characters encoded
          # as <ff> (i.e., inside brackets.)  Does not apply to
          # Keyword/Option pairs, as <> can be PS commands
          if not option and data.find('<') >= 0:
            # expand hex references using the "re" module.
            # (we've precompiled these functions at the end)
            data = _hexre.sub(_hexToBinary, data)



      # Handle any imports
      if keyword == 'Import':
        print "   Importing another file"
        self.loadFile(data)

      elif keyword == 'SymbolValue':
        print "   Setting symbol %s" % option[1:]
        if not self.__symbols__.has_key(option[1:]):
          self.__symbols__[option[1:]] = data

      # Keywords we don't need to keep.
      # We include all "query" and JCL keywords (*?.., *JCL...)
      elif keyword in ('End','SymbolLength','SymbolEnd') \
           or keyword[:1] == '?' or keyword[:3] == 'JCL':
        pass

      elif keyword[:7] == 'Default':
        print "    Setting default value for %s" % keyword[7:]
        # Only keep the first reference... later
        # values do not override the first.
        if not self.__defaults__.has_key(keyword[7:]):
          self.__defaults__[keyword[7:]] = data

      else:
        try:
          cache = self.__properties__[keyword]
        except KeyError:
          cache = []
          self.__properties__[keyword] = cache

        prop = _Property(keyword, option, option_translation, data)
        cache.append(prop)

      # Next in line, please!
      line = handle.readline()

    # If we opened a file, close it.
    if close:
      handle.close()


class _Property:
  def __init__(self, keyword, option=None, option_translation=None, data=None):
    self.keyword = keyword
    self.option = option
    self.option_translation = option_translation
    self.data = data

  def __repr__(self):
    if self.option:
      return "<PrinterDefinition._Property (%s/%s) at %s>" % (self.keyword, self.option, id(self))
    else:
      return "<PrinterDefinition._Property (%s) at %s>" % (self.keyword, id(self))


class _SymbolData:
  def __init__(self, handler, symbol):
    self.handler = handler
    self.symbol = symbol

  def __str__(self):
    try:
      return self.__value__
    except:
      try:
        self.__value__ = val = self.handler.__symbols__[self.symbol]
        return val
      except KeyError:
        raise InvalidPPDFormat, 'Symbol %s referenced, but not defined." % (self.symbol)'



# Used by the class above to convert a
# hex-encoded string to a binary string.
def _hexToBinary( match ):
  s = str(match.group())[1:-1]
  rs = ""
  try:
    for i in range(len(s)/2):
      rs += chr(float('0x' + s[i*2:i*2+2]))
  except ValueError:
    raise InvalidPPDFormat, 'Not a hexadecimal value: <%s>' % s

  return rs

# Precompile our regular expression
_hexre = re.compile(r'<.*?>')



if __name__ == '__main__':
  import sys
  PrinterDefinition(sys.argv[1])
