#if !defined (DEBUG)
#define DEBUG 1
#endif

#include <AppKit.h>
#include <InterfaceKit.h>
#include <Debug.h>

#include <stdio.h>
#include <stdlib.h>
#include "gtk/gtk.h"
#include "libgimp/gimp.h"
#include "libgimp/gimpui.h"
#include "libgimp/gimpprotocol.h"

#include "IEWindow.h"
#include "GimpStub.h"
#include "FilterViews.h"

//-----------------------------------------------------------------------------------------------
//	To port a GIMP filter,
//
//		put whatever is needeed in the constructor for GIMPDlog
//		In the destructor, add whatever is necessary.
//		Update DoPreview
//		Update MessageReceived
//		Update DoIt
//		
//-----------------------------------------------------------------------------------------------
typedef struct
{
  gdouble tile_size;
  gdouble tile_height;
  gdouble tile_spacing;
  gdouble tile_neatness;
  gdouble light_dir;
  gdouble color_variation;
  gint    antialiasing;
  gint    color_averaging;
  gint    tile_type;
  gint    tile_surface;
  gint    grout_color;
} MosaicVals;

extern "C" MosaicVals		mvals;

typedef struct ReturnValue {
	BMessage	*msg;
	bool		done;
};

class GIMPDlog : public IEWindow {
public:
	GIMPDlog (ReturnValue *ret, FILTER_NAME *filter, TFilterParams *parms);
	~GIMPDlog(void);

virtual	void		MessageReceived(BMessage *message);
virtual	bool		QuitRequested (void);
		void		DoPreview (void);
//public:
protected:
	FILTER_NAME			*fFilter;
	TFilterParams		*fParms;
	BSliderTextHelper	*fHelper1, *fHelper2, *fHelper3;
	BSliderTextHelperFloat *fHelper4, *fHelper5;
	ReturnValue			*fReturn;
	bool				fPreview;
//	GIMP fields
	gdouble				tile_size, tile_height, tile_spacing, tile_neatness, light_dir, color_variation;
	gint				antialiasing, color_averaging, tile_type, tile_surface, grout_color;

};

GIMPDlog::GIMPDlog (ReturnValue *ret, FILTER_NAME *filter, TFilterParams *parms) : IEWindow("mosaic") {
	ASSERT (!ret->done); fReturn = ret; fReturn->msg = NULL;
	AddTemporaryBPath (NULL); fParms = parms; fFilter = filter;

	tile_size = mvals.tile_size; tile_height = mvals.tile_height; tile_spacing = mvals.tile_spacing;
	tile_neatness = mvals.tile_neatness; light_dir = mvals.light_dir; color_variation = mvals.color_variation;
	antialiasing = mvals.antialiasing; color_averaging = mvals.color_averaging; tile_type = mvals.tile_type;
	tile_surface = mvals.tile_surface; grout_color = mvals.grout_color;
	fPreview = true;
	Lock ();
	BControl	*ctl;
	//	Handle the check boxes
	if (antialiasing) { ctl = (BControl *) FindView ("Alis"); ASSERT (typeid(*ctl)==typeid(BCheckBox)); ctl->SetValue (1); }
	if (color_averaging) { ctl = (BControl *) FindView ("Colr"); ASSERT (typeid(*ctl)==typeid(BCheckBox)); ctl->SetValue (1); }
	if (tile_surface) { ctl = (BControl *) FindView ("Srfc"); ASSERT (typeid(*ctl)==typeid(BCheckBox)); ctl->SetValue (1); }
	if (grout_color) { ctl = (BControl *) FindView ("Lite"); ASSERT (typeid(*ctl)==typeid(BCheckBox)); ctl->SetValue (1); }
	//	Handle the radios
	char	radioName[2] = { 48, 0 };
	*radioName = 48 + tile_type;
	ctl = (BControl *) FindView (radioName); ASSERT (ctl); if (ctl) ctl->SetValue (1);
	//	Handle the angle
	(new TAngleView (FindView ("Angl"), new BMessage ('Angl')))->SetValue (light_dir);
	//	The sliders
	fHelper1 = new BSliderTextHelper ("Sl1", "TE1", this); fHelper1->SetValue (tile_size);
	fHelper2 = new BSliderTextHelper ("Sl2", "TE2", this); fHelper2->SetValue (tile_height);
	fHelper3 = new BSliderTextHelper ("Sl3", "TE3", this); fHelper3->SetValue (tile_spacing);
	fHelper4 = new BSliderTextHelperFloat ("Sl4", "TE4", this, 100); fHelper4->SetValue (tile_neatness);
	fHelper5 = new BSliderTextHelperFloat ("Sl5", "TE5", this, 100); fHelper5->SetValue (color_variation);
	gProgressView = new TDlogProgressView (FindView ("Progress"));
	Unlock ();
	Show();
	DoPreview ();
}

GIMPDlog::~GIMPDlog(void) {
	if (!fReturn->msg) {
		fParms->fDst->Copy (*fParms->fSrc);
		FilterUpdate (fParms);
	} else  {
		if (!fPreview) {
			fPreview = true;
			DoPreview ();	//	We do the action if it was not previewed.
		}
		VERIFY (B_OK == fFilter->Archive (fReturn->msg, true));
	}
	delete fHelper1; delete fHelper2; delete fHelper3; delete fHelper4; delete fHelper5; 
	fReturn->done = true;
}

bool GIMPDlog::QuitRequested (void) {
	gProgressView = NULL;
	return true;
}

void GIMPDlog::DoPreview (void) {
	if (fPreview) {
		MosaicVals	values;
		values.tile_size = tile_size; values.tile_height = tile_height; values.tile_spacing = tile_spacing;
		values.tile_neatness = tile_neatness; values.light_dir = light_dir; values.color_variation = color_variation;
		values.antialiasing = antialiasing; values.color_averaging = color_averaging; values.tile_type = tile_type;
		values.tile_surface = tile_surface; values.grout_color = grout_color;
		fFilter->SetValue (&values);
	}
}

// Handling of user interface and other events
void GIMPDlog::MessageReceived (BMessage *message) {
	int32			value;

	switch(message->what){
		case '_xPr':	//	DoMiniPreview (); break;
		case 'Sl1 ':	value = message->FindInt32 ("be:value"); if (value != tile_size) { tile_size = value; DoPreview(); } break;
		case 'Sl2 ':	value = message->FindInt32 ("be:value"); if (value != tile_height) { tile_height = value; DoPreview(); } break;
		case 'Sl3 ':	value = message->FindInt32 ("be:value"); if (value != tile_spacing) { tile_spacing = value; DoPreview(); } break;
		case 'Sl4 ':	value = message->FindInt32 ("be:value"); if (value != 100 * tile_neatness) { tile_neatness = float(value) / 100; DoPreview(); } break;
		case 'Sl5 ':	value = message->FindInt32 ("be:value"); if (value != 100 * color_variation) { color_variation = float(value) / 100; DoPreview(); } break;
		case 'Angl':	value = message->FindInt32 ("be:value"); if (value != light_dir) { light_dir = value; DoPreview (); } break;
		case 'Alis':	antialiasing = !antialiasing; DoPreview (); break;
		case 'Colr':	color_averaging = !color_averaging; DoPreview (); break;
		case 'Srfc':	tile_surface = !tile_surface; DoPreview (); break;
		case 'Lite':	grout_color = !grout_color; DoPreview (); break;
		case 'View':	fPreview = !fPreview;
						if (fPreview) {
							DoPreview ();
						} else {
							//	Removes the preview by restoring the original image.
							fParms->fDst->Copy (*fParms->fSrc);
							FilterUpdate (fParms);
						}
						break;
		case 1000:
		case 1001:
		case 1002:		if (int32(message->what - 1000) != tile_type) { tile_type = message->what - 1000; DoPreview (); } break;
		case 'Abut':	fFilter->DoAbout (); break;
		case 'OK  ':	fReturn->msg = new BMessage ('Parm');
						//	Save the parameters
						mvals.tile_size = tile_size; mvals.tile_height = tile_height; mvals.tile_spacing = tile_spacing;
						mvals.tile_neatness = tile_neatness; mvals.light_dir = light_dir; mvals.color_variation = color_variation;
						mvals.antialiasing = antialiasing; mvals.color_averaging = color_averaging; mvals.tile_type = tile_type;
						mvals.tile_surface = tile_surface; mvals.grout_color = grout_color;
		case 'Esc ':	gProgressView = NULL; Quit (); break;
		default:		inherited::MessageReceived(message); break;
	}

}

void FILTER_NAME::DoIt (TFilterParams *parms, const void *values) {
	GParam	inputParms[14], *outputParms;
	gint	cntOutput;

	inputParms[0].type = PARAM_INT32; inputParms[0].data.d_int32 = RUN_NONINTERACTIVE;	//	TBD, determined by the message
	inputParms[2].type = PARAM_DRAWABLE; inputParms[2].data.d_drawable = gint32(parms);
	inputParms[3].type = PARAM_FLOAT; inputParms[3].data.d_float = ((const MosaicVals *)values)->tile_size;
	inputParms[4].type = PARAM_FLOAT; inputParms[4].data.d_float = ((const MosaicVals *)values)->tile_height;
	inputParms[5].type = PARAM_FLOAT; inputParms[5].data.d_float = ((const MosaicVals *)values)->tile_spacing;
	inputParms[6].type = PARAM_FLOAT; inputParms[6].data.d_float = ((const MosaicVals *)values)->tile_neatness;
	inputParms[7].type = PARAM_FLOAT; inputParms[7].data.d_float = ((const MosaicVals *)values)->light_dir;
	inputParms[8].type = PARAM_FLOAT; inputParms[8].data.d_float = ((const MosaicVals *)values)->color_variation;
	inputParms[9].type = PARAM_INT32; inputParms[9].data.d_int32 = ((const MosaicVals *)values)->antialiasing;
	inputParms[10].type = PARAM_INT32; inputParms[10].data.d_int32 = ((const MosaicVals *)values)->color_averaging;
	inputParms[11].type = PARAM_INT32; inputParms[11].data.d_int32 = ((const MosaicVals *)values)->tile_type;
	inputParms[12].type = PARAM_INT32; inputParms[12].data.d_int32 = ((const MosaicVals *)values)->tile_surface;
	inputParms[13].type = PARAM_INT32; inputParms[13].data.d_int32 = ((const MosaicVals *)values)->grout_color;
	(*PLUG_IN_INFO.run_proc) (fName, 14, inputParms, &cntOutput, &outputParms);
}

BMessage *FILTER_NAME::ApplyFilter (TFilterParams *parms) {
	BMessage	*result = NULL;

	if (fInited) {
		ASSERT (fLock == NULL);
		DoIt (parms, fData);
	} else {
		ReturnValue	returnVal;

		fLock = new Benaphore ("Synchro Dialog");
		returnVal.done = fChanged = false; fInited = true;
		AddTemporaryBPath (filterFile);
		new GIMPDlog (&returnVal, this, parms);
		do {
			if (fChanged) {
				fLock->Lock ();
				MosaicVals	tmpValues;
				memcpy (&tmpValues, fData, sizeof (MosaicVals));
				fChanged = false;
				fLock->Unlock ();
				DoIt (parms, &tmpValues);
				FilterUpdate (parms);
			} else {
				snooze (250000);
			}
		} while (!returnVal.done);
		result = returnVal.msg;
	}
	return result;
}

void FILTER_NAME::PrepareData (void) {
	fData = (void *) new char [sizeof(MosaicVals)]; fDataSize = sizeof (MosaicVals);
	ASSERT (fData);
}

void FILTER_NAME::DisposeData (void) {
	delete (MosaicVals *)fData;
}

void FILTER_NAME::SetValue (void *value) {
	ASSERT (fLock);
	fLock->Lock ();

	memcpy (fData, value, sizeof(MosaicVals));
	fChanged = true;
	fLock->Unlock ();
}

void gimp_get_data (gchar *id, gpointer data) {
	ASSERT (gCurFilter); gCurFilter->GetData (id, data);
}

void gimp_set_data (gchar *id, gpointer data, guint32 length) {
	ASSERT (gCurFilter); gCurFilter->SetData (id, data, length);
}

