/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* SETTINGS.C -- Settings manager. */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dir.h>
#include <assert.h>
#include <stdarg.h>
#include <graphics.h>
#include <time.h>
#include "window.h"
#include "settings.h"
#include "canvas.h"
#include "env.h"
#include "resource.h"
#include "key.h"

#define MIN_MOUSE_SPEED         24
#define MIN_PICK_RADIUS         3
#define MIN_DOUBLE_CLICK_DELTA  delta(.17)
#define SET_SUFFIX      ".set"

/* Lazy initializer for dialogs and menus. Breaks
   a mutual dependency with env.c.  Must waits
   until environment initializer has set up menus
   before this will work correctly. */
LOCAL(void) init(void);

/* ----- Options menu ----------------------------------------------- */

static void handle_zoom_constraint(void);
static void handle_move_tools(void);
static void handle_ruler(void);
static void handle_output(void);
static void handle_preferences(void);
static void handle_crosshairs(void);
static void handle_colors(void);

#define config_item_text(I)     (config_menu->entries[I].text.str)

static char tools_left_msg[]  = "Tools left", 
	    tools_right_msg[] = "Tools right";
#define move_tools_msg          config_item_text(0)

static char ruler_on_msg[] = "Ruler on",
		ruler_off_msg[] = "Ruler off";
#define ruler_msg       config_item_text(1)

static char crosshairs_on_msg[] = "Crosshairs on",
		crosshairs_off_msg[] = "Crosshairs off";
#define crosshairs_msg  config_item_text(2)

static char zoom_smooth_msg[] ="Zoom smooth",
		zoom_constrained_msg[] = "Zoom constrained";
#define zoom_constraint_msg     config_item_text(3)

BeginDefMenu(config_menu)
  DefMenuEntry(tools_right_msg,   0, ALT_L, Enabled,       ActionClosure(handle_move_tools, NullEnv))
  DefMenuEntry(ruler_on_msg,      0, ALT_U, Enabled,       ActionClosure(handle_ruler, NullEnv))
  DefMenuEntry(crosshairs_on_msg, 0, ALT_H, Enabled|Ruled, ActionClosure(handle_crosshairs, NullEnv))
  DefMenuEntry(zoom_smooth_msg,   0, ALT_Z, Enabled,       ActionClosure(handle_zoom_constraint, NullEnv))
EndDefMenu(config_menu, 24)

BeginDefMenu(options_menu)
  DefSubmenuEntry("Configuration", 0, config_menu, Enabled)
  DefMenuEntry("Preferences",      0, ALT_N,       Enabled,       ActionClosure(handle_preferences, NullEnv))
  DefMenuEntry("Colors",           2, NoHotKey,    Enabled|Ruled, ActionClosure(handle_colors, NullEnv))
  DefMenuEntry("Output",           0, ALT_T,       Enabled,       ActionClosure(handle_output, NullEnv))
EndDefMenu(options_menu, 24)

LOCAL(void) menu2settings(void)
{
  set_bit(settings->status, sZOOM_SMOOTH,   zoom_constraint_msg == zoom_constrained_msg);
  set_bit(settings->status, sTOOLS_RIGHT,   move_tools_msg == tools_left_msg);
  set_bit(settings->status, sRULER_ON,      ruler_msg == ruler_off_msg);
  set_bit(settings->status, sCROSSHAIRS_ON, crosshairs_msg == crosshairs_off_msg);
}

void settings2menu(void)
{
  init();
  zoom_constraint_msg = 
    settings->status & bit(sZOOM_SMOOTH) ? 
      zoom_constrained_msg :
      zoom_smooth_msg;

  move_tools_msg =
    settings->status & bit(sTOOLS_RIGHT) ?
      tools_left_msg :
      tools_right_msg;

  ruler_msg =
    settings->status & bit(sRULER_ON) ?
      ruler_off_msg :
      ruler_on_msg;

  crosshairs_msg = 
    settings->status & bit(sCROSSHAIRS_ON) ?
      crosshairs_off_msg :
      crosshairs_on_msg;
}

static void handle_zoom_constraint(void)
{
  if (zoom_constraint_msg == zoom_smooth_msg) {
    settings->status |= bit(sZOOM_SMOOTH);
    zoom_constraint_msg = zoom_constrained_msg;
  }
  else {
    settings->status &= notbit(sZOOM_SMOOTH);
    zoom_constraint_msg = zoom_smooth_msg;
  }
  menu2settings();
}

static void handle_move_tools(void)
{
  if (move_tools_msg == tools_right_msg) {
    /* move right as promised by menu entry */
    move_tools(1);
    move_tools_msg = tools_left_msg;
    settings->status |= bit(sTOOLS_RIGHT);
  }
  else {
    /* move left */
	move_tools(0);
    move_tools_msg = tools_right_msg;
    settings->status &= notbit(sTOOLS_RIGHT);
  }
  menu2settings();
}

static void handle_ruler(void)
{
  if (ruler_msg == ruler_off_msg) {
    disable_canvas_ruler(edit_canvas);
    ruler_msg = ruler_on_msg;
  }
  else {
    enable_canvas_ruler(edit_canvas);
    ruler_msg = ruler_off_msg;
  }
  menu2settings();
}

/* Call vset_cursor with translation of mask to substitute 
   mouse pointer for hairs if user wants hair off. */
LOCAL(void) do_set_cursor(unsigned mask, va_list ap)
{
  vset_cursor(&edit_canvas->cursor, 
	      !(settings->status & bit(sCROSSHAIRS_ON)) 
		&& (mask & bit(cHAIRS)) ? 
		  (mask & notbit(cHAIRS)) | bit(cMOUSE) 
		  : mask, 
	      ap);
}

static unsigned current_mask;

/* Set cursor accounting for setting to suppress crosshairs. */
void env_set_cursor(unsigned mask, ...)
{
  va_list ap;

  va_start(ap, mask);
  do_set_cursor(current_mask = mask, ap);
  va_end(ap);
}

static void handle_crosshairs(void)
{
  crosshairs_msg = (crosshairs_msg == crosshairs_off_msg) ? 
			crosshairs_on_msg : crosshairs_off_msg;
  menu2settings();
  do_set_cursor(current_mask, NULL);
}

/* ----- Output dialog ---------------------------------------------- */

SETTINGS_REC settings[1] = {{ NONAME }};

/* Caption must be in malloc'ed store. */
void set_caption(SETTINGS s, char *caption)
{
  if (s->caption != NULL)
    free(s->caption);
  s->caption = caption;
}

/* Unitlength must be in malloc'ed store. */
void set_unitlength(SETTINGS s, char *unitlength)
{
  if (s->unitlength != NULL)
	free(s->unitlength);
  s->unitlength = unitlength;
}

void handle_picture_checks(int rtn, ENV env);
void handle_units_radio(int pos, ENV env);
static void handle_pick_radius_scroll(int pos, ENV env);
static void handle_mouse_speed_scroll(int pos, ENV env);
static void handle_double_click_scroll(int pos, ENV env);

/* Base unit length for conversions. */
static SP dialog_unitlength;

#define MAX_UNITLENGTH_LEN 15

static void validate_unitlength_edit(char *str, ENV env);

BeginDefEditBox(unitlength_edit_box)
  DefValidator(NameActionClosure(validate_unitlength_edit, NullEnv))
  DefKeyAction('\r', NameActionClosure(surrender_focus_action, unitlength_edit_box))
EndDefEditBox(unitlength_edit_box, "unitlength", 1, 11, MAX_UNITLENGTH_LEN, Enabled)

#define MAX_CAPTION_LEN 200

static void validate_caption_edit(char *str, ENV env);

BeginDefEditBox(caption_edit_box)
  DefValidator(NameActionClosure(validate_caption_edit, NullEnv))
  DefKeyAction('\r', NameActionClosure(surrender_focus_action, caption_edit_box))
EndDefEditBox(caption_edit_box, "caption", 5, 25, MAX_CAPTION_LEN, 0)

BeginDefChecks(picture_checks)
  DefChecksItem("unitlength", 0)
  DefChecksItem("figure", 0)
  DefChecksItem("center", 5)
  DefChecksItem("caption", 1)
  DefChecksItem("document", 0)
EndDefChecks(picture_checks, "Picture", 0, Enabled, PositionActionClosure(handle_picture_checks, NullEnv));

BeginDefRadio(units_radio)
  DefRadioItem("cm ", 0)        /* Extra space to make box wider. */
  DefRadioItem("em", 0)
  DefRadioItem("ex", 1)
  DefRadioItem("in", 0)
  DefRadioItem("mm", 0)
  DefRadioItem("pc", 0)
  DefRadioItem("pt", 1)
EndDefRadio(units_radio, "Units", 0, Enabled, PositionActionClosure(handle_units_radio, NullEnv));

extern DIALOG_REC output_dialog[];

static void handle_output_button(int *ok_p);
DefBitmapButton(output_ok,     ok,     'k', 2, 2, Enabled, ActionClosure(handle_output_button, val_one))
DefBitmapButton(output_cancel, cancel, ESC, 2, 2, Enabled, ActionClosure(handle_output_button, val_zero))

BeginDefDialog(output_dialog)
  DefDialogItem(CheckBoxItem, picture_checks, 8, 8)
  DefDialogItem(EditBoxItem, unitlength_edit_box, 120, 26)
  DefDialogItem(EditBoxItem, caption_edit_box, 8, 107)
  DefDialogItem(RadioItem, units_radio, 250, 16)
  DefDialogItem(ButtonItem, output_ok, 217, 49)
  DefDialogItem(ButtonItem, output_cancel, 203, 78)
EndDefDialog(output_dialog, "Output options", Enabled)

#pragma argsused

static void handle_picture_checks(int rtn, ENV env)
{
  if (check_rtn_pos(rtn) == sCAPTION)
    (check_rtn_set(rtn) ? enable_edit_box : disable_edit_box)(caption_edit_box);
}

#pragma argsused 

static void handle_units_radio(int pos, ENV env)
{
  char buf[32];

  TeXlength2str(dialog_unitlength, buf, pos); 
  set_edit_text(unitlength_edit_box, buf);
}

LOCAL(int) check_unitlength_edit(char *unitlength_str)
{
  UNITS_TYPE units;
  static int already_checking_p = 0;
  int rtn;

  /* Have to prevent a recursive check when message box
     causes focus to be withdrawn from edit. Another approach
     would be to put a legal string in the edit before
     invoking the message box. */
  if (already_checking_p)
    return 1;
  already_checking_p = 1;

  if (str2TeXlength(unitlength_str, &dialog_unitlength, (UNITS_TYPE*)&units)) {
    set_radio_pos(units_radio, units);
    rtn = 1;
  }
  else {
    message(bit(mOK), &output_dialog->window, 
	    "`%s' is not a unit length.\n"
	    "Examples: 1mm  .05in  10pt", unitlength_str);
    set_focus(&unitlength_edit_box->window);
    rtn = 0;
  }
  already_checking_p = 0;
  return rtn;
}

#pragma argsused 

static void validate_unitlength_edit(char *str, ENV env)
{
  check_unitlength_edit(str);
}

LOCAL(int) check_caption_edit(char *text, unsigned msg_mask)
{
  int reply, rtn;
  static int already_checking_p = 0;

  /* Have to prevent a recursive check when message box
     causes focus to be withdrawn from edit. Another approach
     would be to put a legal string in the edit before
     invoking the message box. */
  if (already_checking_p)
    return 1;
  already_checking_p = 1;

  if (text_is_latext(text)) {
    rtn = 1;
  }
  else {
    reply = message(msg_mask, &output_dialog->window, 
		    "Caption is questionable\n"
		    "LaTeX.  It will cause\n"
		    "File Open or Merge to\n"
		    "fail later.\n%s",

		    (msg_mask & bit(mYES)) ? 
		    " \n"
		    "Want to accept anyway?" 
		    : "");

    if (reply == mYES) 
      rtn = 1;
    else {
      set_focus(&caption_edit_box->window);
      rtn = 0;
    }
  }
  already_checking_p = 0;
  return rtn;
}

#pragma argsused

static void validate_caption_edit(char *edit_str, ENV env)
{ 
  check_caption_edit(edit_str, bit(mOK));
}

#pragma argsused 

static void handle_output_button(int *ok_p)
{
  if (!*ok_p || 
      (check_unitlength_edit(get_edit_text(unitlength_edit_box)) &&
       check_caption_edit(get_edit_text(caption_edit_box), bit(mYES)|bit(mNO))))
    stop_modal_dialog(output_dialog, *ok_p);
}

/* Transfer contents of global settings to the output dialog. */
void settings2output_dialog(SETTINGS s)
{
  init();
  set_checks_on_mask(picture_checks, s_read_settings(settings));
  set_edit_text(unitlength_edit_box, s->unitlength);
  str2TeXlength(s->unitlength, &dialog_unitlength, NULL);
  set_edit_text(caption_edit_box, s->caption);
  (s->status & bit(sCAPTION) ? enable_edit_box : disable_edit_box)(caption_edit_box);
  set_radio_pos(units_radio, s->units);
}

/* Transfer contents of output dialog to global settings. */
LOCAL(void) output_dialog2settings(void)
{
  char *str;

  set_read_settings(settings, checks_on_mask(picture_checks));
  str = get_edit_text(unitlength_edit_box);
  assert(str2TeXlength(str, NULL, NULL));
  set_unitlength(settings, strdup(str));
  set_canvas_unitlength(edit_canvas, dialog_unitlength);
  str = get_edit_text(caption_edit_box);
  if (settings->status & bit(sCAPTION))
    set_caption(settings, strdup(str));
  settings->units = radio_pos(units_radio);
}

static void handle_output(void)
{
  if (run_modal_dialog(output_dialog)) 
    output_dialog2settings();
  else
    settings2output_dialog(settings);
}

/* ----- Preferences dialog ----------------------------------------- */

BeginDefChecks(file_checks)
  DefChecksItem("Auto-load", 0)
  DefChecksItem("Backup", 0)
EndDefChecks(file_checks, "File", 0, Enabled, NullPositionAction)

BeginDefChecks(pick_checks)
  DefChecksItem("Flash", 0)
  DefChecksItem("Off-screen warn", 0)
  DefChecksItem("Auto-displacement", 1)
EndDefChecks(pick_checks, "Picks", 0, Enabled, NullPositionAction)

#define DOUBLE_CLICK_SCROLL_NPOS        delta(.6)
DefScrollbar(pick_radius_scroll, "Pick diameter",0, 100, 16, 1,  1, 0, Enabled,         PositionActionClosure(handle_pick_radius_scroll, NullEnv))
DefScrollbar(mouse_speed_scroll, "Mouse speed",  0, 100, 64, 1,  8, 0, Enabled,         PositionActionClosure(handle_mouse_speed_scroll, NullEnv))
DefScrollbar(double_click_scroll,"Double-click", 0, 100, DOUBLE_CLICK_SCROLL_NPOS, DOUBLE_CLICK_SCROLL_NPOS/10,  DOUBLE_CLICK_SCROLL_NPOS/10, DOUBLE_CLICK_SCROLL_NPOS - 1, Enabled|Reverse, PositionActionClosure(handle_double_click_scroll, NullEnv))

extern DIALOG_REC preferences_dialog[];

static void handle_preferences_button(int *ok_p);
DefBitmapButton(preferences_ok,     ok,     'k', 2, 2, Enabled, ActionClosure(handle_preferences_button, val_one))
DefBitmapButton(preferences_cancel, cancel, ESC, 2, 2, Enabled, ActionClosure(handle_preferences_button, val_zero))

static void validate_auto_redraw_edit(char *str, ENV env);

BeginDefEditBox(auto_redraw_edit_box)
  DefValidator(NameActionClosure(validate_auto_redraw_edit, NullEnv))
  DefKeyAction('\r', NameActionClosure(surrender_focus_action, auto_redraw_edit_box))
EndDefEditBox(auto_redraw_edit_box, "Redraw", 0, 4, 4, Enabled)

BeginDefDialog(preferences_dialog)
  DefDialogItem(CheckBoxItem, file_checks, 8, 8)
  DefDialogItem(CheckBoxItem, pick_checks, 8, 56)
  DefDialogItem(ScrollbarItem, pick_radius_scroll, 8, 129)
  DefDialogItem(ScrollbarItem, mouse_speed_scroll, 112, 16)
  DefDialogItem(ScrollbarItem, double_click_scroll, 112, 41)
  DefDialogItem(EditBoxItem, auto_redraw_edit_box, 178, 66)
  DefDialogItem(ButtonItem, preferences_ok, 184, 119)
  DefDialogItem(ButtonItem, preferences_cancel, 216, 119)
EndDefDialog(preferences_dialog, "Environment options", Enabled)

static_WINDOW(pick_radius_picture);
static_WINDOW(double_click_test);

static int pick_radius_picture_rad = MIN_PICK_RADIUS;

/* Draw a circle with arrow dimension marker
   in the pick radius picture window. */
static void handle_pick_radius_picture_map(EVENT e)
{
  int r, cx, cy, x1, x2, x3, x4;
  WINDOW w;

  w = e->map.window;
  push_graphics_state(w, 1);
  protect_cursor(w);
  setcolor(cBLACK);
  r = pick_radius_picture_rad;
  cx = w->width/2;
  cy = w->height/2;
  circle(cx, cy, r);
  x2 = cx - r;
  x3 = cx + r;
  if (r < 8) {
    x1 = x2 - 8;
    x4 = x3 + 8;
    arrowhead(x1, cy, x2, cy);
    line(x1, cy, x2, cy);
    arrowhead(x4, cy, x3, cy);
    line(x4, cy, x3, cy);
  }
  else {
    arrowhead(x2, cy, x3, cy);
    arrowhead(x3, cy, x2, cy);
    line(x2, cy, x3, cy);
  }
  unprotect_cursor();
  pop_graphics_state();
}

static int double_click_test_reverse_p;

LOCAL(void) draw_double_click_test_msg(WINDOW w)
{
  int bg_color, fg_color;

  if (double_click_test_reverse_p) {
    bg_color = settings->color[ccTHIN];
    fg_color = settings->color[ccBACKGROUND];
  }
  else {
    fg_color = settings->color[ccTHIN];
    bg_color = settings->color[ccBACKGROUND];
  }
  set_window_bg_color(w, bg_color, 0);

  push_graphics_state(w, 0);
  protect_cursor(w);
  setcolor(fg_color);
  settextstyle(SMALL_FONT, HORIZ_DIR, 4);
  settextjustify(CENTER_TEXT, CENTER_TEXT);
  outtextxy(w->width/2, w->height/2 - 1, "TEST");
  unprotect_cursor();
  pop_graphics_state();  
}

LOCAL(void) handle_double_click_button(EVENT e)
{
  if (double_click_p(e->mouse.window)) {
    double_click_test_reverse_p = 1 - double_click_test_reverse_p;
    draw_double_click_test_msg(e->mouse.window);
  }
}

BeginDefDispatch(double_click_test)
  DispatchAction(eMAP, draw_double_click_test_msg(e->map.window))
  Dispatch(eBUTTON_PRESS, handle_double_click_button)
EndDefDispatch(double_click_test)

#pragma argsused

static void handle_pick_radius_scroll(int pos, ENV env)
{
  pick_radius_picture_rad = MIN_PICK_RADIUS + pos;
  clear_window(pick_radius_picture, 1);
}

#pragma argsused

static void handle_mouse_speed_scroll(int pos, ENV env)
{
  set_mouse_speed(MIN_MOUSE_SPEED + pos);
}

#pragma argsused

static void handle_double_click_scroll(int pos, ENV env)
{
  set_double_click_delta(MIN_DOUBLE_CLICK_DELTA + pos);
}

LOCAL(int) check_auto_redraw_edit(char *redraw_str)
{
  int dummy, rtn;
  static int already_checking_p = 0;

  /* Have to prevent a recursive check when message box
     causes focus to be withdrawn from edit. Another approach
     would be to put a legal string in the edit before
     invoking the message box. */
  if (already_checking_p)
    return 1;
  already_checking_p = 1;

  if (sscanf(redraw_str, "%d", &dummy) == 1 && dummy >= 0) {
    rtn = 1;
  }
  else {
    message(bit(mOK), &preferences_dialog->window, 
	    "`%s' is not an auto\n" 
	    "redraw threshhold.\n"
	    "Example: 200",
	    redraw_str);
    set_focus(&auto_redraw_edit_box->window);
    rtn = 0;
  }
  already_checking_p = 0;
  return rtn;
}

#pragma argsused

static void validate_auto_redraw_edit(char *str, ENV env)
{
  check_auto_redraw_edit(str);
}

static void handle_preferences_button(int *ok_p)
{
  if (!*ok_p || check_auto_redraw_edit(get_edit_text(auto_redraw_edit_box)))
    stop_modal_dialog(preferences_dialog, *ok_p);
}

/* Transfer contents of global settings to the preferences dialog. */
LOCAL(void) settings2preferences_dialog(void)
{
  char buf[8];
 
  init();
  position_scrollbar(pick_radius_scroll, settings->pick_radius - MIN_PICK_RADIUS);
  pick_radius_picture_rad = settings->pick_radius;
  position_scrollbar(mouse_speed_scroll, settings->mouse_speed - MIN_MOUSE_SPEED);
  set_mouse_speed(settings->mouse_speed);
  set_double_click_delta(settings->double_click_delta);
  position_scrollbar(double_click_scroll, settings->double_click_delta - MIN_DOUBLE_CLICK_DELTA);
  set_checks_on_mask(file_checks, s_file_settings(settings));
  set_checks_on_mask(pick_checks, s_pick_settings(settings));
  set_edit_text(auto_redraw_edit_box, itoa(settings->auto_redraw, buf, 10));
}

/* Transfer contents of output dialog to global settings. 
   Also adjust the physical preferences. */
LOCAL(void) preferences_dialog2settings(void)
{
  settings->pick_radius = pick_radius_picture_rad;
  settings->double_click_delta = get_double_click_delta();
  settings->mouse_speed = get_mouse_speed();
  set_file_settings(settings, checks_on_mask(file_checks));
  set_pick_settings(settings, checks_on_mask(pick_checks));
  if (settings->status & bit(sFLASHING_PICK)) 
    enable_flashing_pick(edit_canvas);
  else
    disable_flashing_pick();
  settings->auto_redraw = atoi(get_edit_text(auto_redraw_edit_box));
  force_canvas_change(edit_canvas);
}

/* Handle preferences menu selection. */
static void handle_preferences(void)
{
  if (run_modal_dialog(preferences_dialog)) 
    preferences_dialog2settings();
  else
    settings2preferences_dialog();
}

/* ----- Color menu ------------------------------------------------- */

static void handle_color_select(ENV env);

BeginDefColorSelector(colors_selector, 1)
  DefColorSelectorButton(colors_selector, 0, cBLACK,     Enabled, handle_color_select)
  DefColorSelectorButton(colors_selector, 1, cWHITE,     Enabled, handle_color_select)
  DefColorSelectorButton(colors_selector, 2, cDARKGRAY,  Enabled, handle_color_select)
  DefColorSelectorButton(colors_selector, 3, cLIGHTGRAY, Enabled, handle_color_select)
  DefColorSelectorButton(colors_selector, 4, cBLUE,      Enabled, handle_color_select)
  DefColorSelectorButton(colors_selector, 5, cLIGHTBLUE, Enabled, handle_color_select)
  DefColorSelectorButton(colors_selector, 6, cCYAN,      Enabled, handle_color_select)
  DefColorSelectorButton(colors_selector, 7, cLIGHTCYAN, Enabled, handle_color_select)
  DefColorSelectorButton(colors_selector, 8, cRED,       Enabled, handle_color_select)
  DefColorSelectorButton(colors_selector, 9, cLIGHTRED,  Enabled, handle_color_select)
  DefColorSelectorButton(colors_selector,10, cGREEN,     Enabled, handle_color_select)
  DefColorSelectorButton(colors_selector,11, cLIGHTGREEN,Enabled, handle_color_select)
EndDefColorSelector(colors_selector, Vertical|Enabled)

LOCAL(BUTTON) color2button(int color)
{
  int i;
  BUTTON p;

  for (i = 0, p = colors_selector_buttons; i < array_size(colors_selector_buttons); ++i, ++p)
    if (color == p->fg_color)
      return p;
  return NULL;
}

extern DIALOG_REC colors_dialog[];
static void settings2colors_dialog(void);
static void handle_reset_button(void);

static void handle_colors_button(int *ok_p);
DefBitmapButton(colors_ok,     ok,     'k', 2, 2, Enabled, ActionClosure(handle_colors_button, val_one))
DefBitmapButton(colors_cancel, cancel, ESC, 2, 2, Enabled, ActionClosure(handle_colors_button, val_zero))
DefTextButton(colors_current, "Reset current", 7, NoHotKey, 0, 0, 2, 1, Enabled, ActionClosure(settings2colors_dialog, NullEnv))
DefTextButton(colors_default, "Reset default", 7, NoHotKey, 0, 0, 2, 1, Enabled, ActionClosure(handle_reset_button, NullEnv))

static void handle_colors_radio(int pos, ENV env);

BeginDefRadio(colors_radio)
  DefRadioItem("Background", 0)
  DefRadioItem("Thin lines", 3)
  DefRadioItem("Thick lines", 3)
  DefRadioItem("Pick", 0)
  DefRadioItem("Ruler/origin", 0)
  DefRadioItem("Crosshairs", 2)
EndDefRadio(colors_radio, "Set", 0, Enabled, PositionActionClosure(handle_colors_radio, NullEnv));

BeginDefDialog(colors_dialog)
  DefDialogItem(RadioItem, colors_radio, 8, 8)
  DefDialogItem(ButtonItem, colors_current, 8, 112)
  DefDialogItem(ButtonItem, colors_default, 8, 142)
  DefDialogItem(SelectorItem, colors_selector, 133, 8)
  DefDialogItem(ButtonItem, colors_ok, 173, 141)
  DefDialogItem(ButtonItem, colors_cancel, 201, 141)
EndDefDialog(colors_dialog, "Color options", Enabled)

static_WINDOW(colors_show);

static CANVAS_COLOR_VECTOR colors_box_color = { cBLUE, cLIGHTCYAN, cWHITE, cLIGHTRED, cCYAN, cLIGHTGRAY };
static CANVAS_COLOR_VECTOR default_color    = { cBLUE, cLIGHTCYAN, cWHITE, cLIGHTRED, cCYAN, cLIGHTGRAY };
static char *note[ccN] = { "", "Thin lines", "Thick lines", "Pick", "Ruler/origin", "Crosshairs" };

LOCAL(int) color2box(int color)
{
  int i;
  for (i = 0; i < ccN; ++i)
    if (colors_box_color[i] == color)
      return i;
  return -1;
}

enum { colors_box_x_margin = 4, colors_box_y_margin = 5, colors_box_box_width = 72, };

LOCAL(void) adjust_colors_dialog(void)
{
  feign_button_press(color2button(colors_box_color[radio_pos(colors_radio)]));
}

#pragma argsused
static void handle_colors_radio(int pos, ENV env) 
{
  adjust_colors_dialog();
}

LOCAL(void) draw_color_box(char *text, int y, int color)
{
  int hht;

  setcolor(color);
  hht = textheight(text)/2 + colors_box_y_margin;
  rectangle(colors_box_x_margin, y - hht, colors_box_x_margin + colors_box_box_width, y + hht);
  outtextxy(colors_box_x_margin + colors_box_box_width/2, y, text);
}

/* Draw boxes with current colors. */
static void draw_colors_box(WINDOW w)
{
  int i, y, dy;

  if (!w_status_p(w, wsVISIBLE))
    return;
  set_window_bg_color(w, colors_box_color[ccBACKGROUND], 0);
  push_graphics_state(w, 1);
  protect_cursor(w);
  settextstyle(SMALL_FONT, HORIZ_DIR, 4);
  settextjustify(CENTER_TEXT, CENTER_TEXT);
  dy = textheight(note[0]) + 3 * colors_box_y_margin;
  for (i = 1, y = (dy + colors_box_y_margin)/2; i < ccN; ++i, y += dy)
    draw_color_box(note[i], y, colors_box_color[i]);
  settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
  unprotect_cursor();
  pop_graphics_state();
}

static void handle_color_select(ENV env)
{
  int color, box;

  color = ((BUTTON)env)->fg_color;
  /* If color selected by user matches a color already
     in use, swap to avoid a match. */
  if ((box = color2box(color)) == -1)
    colors_box_color[radio_pos(colors_radio)] = color;
  else
    swap(&colors_box_color[radio_pos(colors_radio)], &colors_box_color[box]);
  draw_colors_box(colors_show);
}

/* Draw boxes with current colors. */
static void handle_colors_box_map(EVENT e)
{
  draw_colors_box(e->map.window);
}

static void handle_colors_button(int *ok_p)
{
  stop_modal_dialog(colors_dialog, *ok_p);
}

static void handle_reset_button(void)
{
  memcpy(settings->color, default_color, sizeof(COLOR_VECTOR));
  settings2colors_dialog();
}

LOCAL(void) colors_dialog2settings(void)
{
  memcpy(settings->color, colors_box_color, sizeof(COLOR_VECTOR));
  set_canvas_colors(edit_canvas, settings->color);
}

static void settings2colors_dialog(void)
{
  init();
  memcpy(colors_box_color, settings->color, sizeof(COLOR_VECTOR));
  adjust_colors_dialog();
}

/* Handle colors menu selection. */
static void handle_colors(void)
{
  if (run_modal_dialog(colors_dialog)) 
    colors_dialog2settings();
  else
    settings2colors_dialog();
}

/* ----- Initialize ------------------------------------------------- */

/* DEBUG:  Is there a DOS function to do this? */
LOCAL(void) justify_path(char *path, char *drive, char *dir, char *name, char *ext)
{
  char buf[MAXPATH];
  int code;

  getcwd(buf, sizeof buf);
  strcat(buf, "\\");
  code = fnsplit(path, drive, dir, name, ext);
  fnsplit(buf, (code & DRIVE) ? NULL : drive, 
	       (code & DIRECTORY) ? NULL : dir, 
	       NULL, NULL);
}

LOCAL(void) unjustify_path(char *path, char *drive, char *dir, char *name, char *ext)
{
  char cur_drive[MAXDRIVE], cur_dir[MAXDIR], buf[MAXPATH];

  getcwd(buf, sizeof buf);
  strcat(buf, "\\");
  fnsplit(buf, cur_drive, cur_dir, NULL, NULL);
  if (stricmp(drive, cur_drive) == 0)
    drive = "";
  if (stricmp(cur_dir, dir) == 0)
    dir = "";
  fnmerge(path, drive, dir, name, ext);
}

LOCAL(int) _fget_line(char *buf, int buf_size, FILE *f)
{
  int ch, i;

  i = 0;
  while ((ch = fgetc(f)) != EOF && ch != '\n')
    if (i < buf_size)
      buf[i++] = ch;
  buf[i] = '\0';
  return (ch != EOF || i != 0);
}

#define fgetln(F, B)    _fget_line(B, sizeof B - 1, F)

LOCAL(int) fget_int_vector(FILE *f, int *v, int n)
{
  int i;

  for (i = 0; i < n; ++i) 
    if (fscanf(f, "%d", &v[i]) != 1)
      return 0;
  return 1;
}

LOCAL(void) read_settings(SETTINGS s, char *path)
{
  int pick_radius, mouse_speed, delta, auto_redraw;
  FILE *f;
  UNITS_TYPE units;
  unsigned status;
  COLOR_VECTOR colors;
  char caption[MAX_CAPTION_LEN + 1], unitlength[MAX_UNITLENGTH_LEN + 1],
       drive[MAXDRIVE], dir[MAXDIR], name[MAXFILE], ext[MAXEXT];

  if ((f = fopen(path, "r")) == NULL)
    return;

  if (string(f, "drive:")       && fgetln(f, drive)
   && string(f, "dir:")         && fgetln(f, dir)
   && string(f, "name:")        && fgetln(f, name)
   && string(f, "ext:")         && fgetln(f, ext))
    unjustify_path(settings->edit_path, drive, dir, name, ext);

  if (string(f, "status:")      && fscanf(f, "%x", &status) == 1
   && string(f, "units:")       && fscanf(f, "%d", &units) == 1
   && 0 <= units && units < uN
   && string(f, "pickradius:")  && fscanf(f, "%d", &pick_radius) == 1
   && 0 <= pick_radius
   && string(f, "mousespeed:")  && fscanf(f, "%d", &mouse_speed) == 1
   && string(f, "dclickdelta:") && fscanf(f, "%d", &delta) == 1
   && string(f, "caption:")     && fgetln(f, caption)
   && string(f, "unitlength:")  && fgetln(f, unitlength)
   && str2TeXlength(unitlength, NULL, NULL)
   && string(f, "redraw:")      && fscanf(f, "%d", &auto_redraw) == 1
   && string(f, "colors:")      && fget_int_vector(f, colors, ccN)) {

    s->status = status;
    if (s->caption != NULL)
      free(s->caption);
    s->caption = strdup(caption);
    if (s->unitlength != NULL) 
      free(s->unitlength);
    s->unitlength = strdup(unitlength);
    s->units = units;
    s->mouse_speed = mouse_speed;
    s->pick_radius = pick_radius;
    s->double_click_delta = delta;
    s->auto_redraw = auto_redraw;
    memcpy(s->color, colors, sizeof(COLOR_VECTOR));
  }
  fclose(f);
}

/* Convert null string pointer to null string. */
#define stringify(S) ((S) ? (S) : "")

LOCAL(void) fprint_int_vector(FILE *f, int *v, int n)
{
  int i;

  for (i = 0; i < n; ++i)
    fprintf(f, " %d", v[i]);
  fprintf(f, "\n");
}

LOCAL(void) write_settings(SETTINGS s, char *path)
{
  FILE *f;
  char drive[MAXDRIVE], dir[MAXDIR], name[MAXFILE], ext[MAXEXT];

  if ((f = fopen(path, "w")) == NULL)
    return;
  justify_path(s->edit_path, drive, dir, name, ext);
  fprintf(f, 

	  "drive:%s\n"                  /* 1 */
	  "dir:%s\n"                    /* 2 */
	  "name:%s\n"                   /* 3 */
	  "ext:%s\n"                    /* 4 */
	  "status:%x\n"                 /* 5 */
	  "units:%d\n"                  /* 6 */
	  "pickradius:%d\n"             /* 7 */
	  "mousespeed:%d\n"             /* 8 */
	  "dclickdelta:%d\n"            /* 9 */
	  "caption:%s\n"                /* 10 */
	  "unitlength:%s\n"             /* 11 */
	  "redraw:%d\n"                 /* 12 */
	  "colors:",                    /* 13 */

	  drive,                        /* 1 */
	  dir,                          /* 2 */
	  name,                         /* 3 */
	  ext,                          /* 4 */
	  s->status,                    /* 5 */
	  s->units,                     /* 6 */
	  s->pick_radius,               /* 7 */
	  s->mouse_speed,               /* 8 */
	  s->double_click_delta,        /* 9 */
	  stringify(s->caption),        /* 10 */
	  stringify(s->unitlength),     /* 11 */
	  s->auto_redraw);              /* 12 */
  fprint_int_vector(f, s->color, ccN);  /* 13 */
  fclose(f);
}

LOCAL(void) set_initial_file(char *raw_path)
{
  int mask;

  char path[MAXPATH], drive[MAXDRIVE], dir[MAXDIR], 
       name[MAXFILE], ext[MAXEXT];
  struct ffblk ffblk;

  /* User asked for file. Add default extension. */
  mask = fnsplit(raw_path, drive, dir, name, ext);
  if ( !(mask & EXTENSION))
    strcpy(ext, ".tex");
  fnmerge(path, drive, dir, name, ext);
  strupr(path);

  /* Look for first file satisfying path.
     If user used wildcards, substitute first match. */
  if (!findfirst(path, &ffblk, 0)) {
    fnsplit(ffblk.ff_name, NULL, NULL, name, ext);
    fnmerge(path, drive, dir, name, ext);
    set_edit_path(settings, path);
  }
  else {
    /* Truncate pathname so message isn't too long. */
    strcpy(path + 40, " ...");
    message(bit(mOK), root_window, "Can't open\n`%s'\nfor input.", path);
  }
}

LOCAL(void) init()
{
  if (static_resource_opened_p(output_dialog))
    return;

  /* Init all dialogs and menus. */
  open_dialog(output_dialog, CENTER_WINDOW, CENTER_WINDOW, root_window);
  open_dialog(colors_dialog, CENTER_WINDOW, CENTER_WINDOW, root_window);
  open_dialog(preferences_dialog, CENTER_WINDOW, CENTER_WINDOW, root_window);
  open_window(pick_radius_picture, &preferences_dialog->window, 116, window_y(&pick_radius_scroll->window) - 11, 41, 21, 2, cBLACK, cWHITE, bit(eMAP));
  open_window(double_click_test,   &preferences_dialog->window, 
	      window_x(&double_click_scroll->window) + double_click_scroll->window.width + 8, 
	      window_y(&double_click_scroll->window) - 7, 
	      31, 17, 
	      2, cBLACK, cWHITE, bit(eMAP)|bit(eBUTTON_PRESS));
  open_window(colors_show, &colors_dialog->window, 
	      156, 9 + header_bar_height(),
	      2*colors_box_x_margin + colors_box_box_width, ccN * 20,
	      2, cBLACK, cWHITE, bit(eMAP));
  map_window(pick_radius_picture);
  pick_radius_picture->handler = handle_pick_radius_picture_map;
  map_window(double_click_test);
  map_window(colors_show);
  colors_show->handler = handle_colors_box_map;
  SetDispatch(double_click_test, double_click_test);
}

/* Initialize global settings and
   preferences from stored settings. */
void init_settings(char *initial_file, char *settings_path)
{
  char buf[MAXPATH];

  settings->status = 0;
  settings->caption = strdup("");
  settings->unitlength = strdup("1mm");
  settings->units = uMM;
  settings->pick_radius = MIN_PICK_RADIUS;
  settings->double_click_delta = MIN_DOUBLE_CLICK_DELTA;
  settings->mouse_speed = MIN_MOUSE_SPEED;
  memcpy(settings->color, default_color, sizeof(COLOR_VECTOR));

  read_settings(settings, change_suffix(settings_path, SET_SUFFIX, buf));
  settings2output_dialog(settings);
  if (initial_file != NULL) 
    set_initial_file(initial_file);
  else if (!(settings->status & bit(sAUTO_LOAD)))
    set_edit_path(settings, NONAME);
  settings2menu();
  settings2preferences_dialog();
  settings2colors_dialog();
}

/* Write the settings to the settings file. */
void close_settings(char *settings_path)
{
  char buf[MAXPATH];

  write_settings(settings, change_suffix(settings_path, SET_SUFFIX, buf));
}
