# Menu creation procs
	
namespace eval menu {}
namespace eval global {}
namespace eval file {}

proc menu::buildBasic {} {
    global winMenu HOME
    # These are built on the fly
    Menu -n File -p menu::generalProc {}
    Menu -n Edit -p menu::generalProc {}
    Menu -n Text -p menu::generalProc {}
    Menu -n Search {}
    Menu -n Utils {}
    Menu -n Config {}
    Menu -n $winMenu {}
    
    insertMenu "File"
    insertMenu "Edit"
    insertMenu "Text"
    insertMenu "Search"
    insertMenu "Utils"
    insertMenu "Config"
    insertMenu $winMenu
    
    if {![catch {glob -dir [file join $HOME Help] *} files]} {
	set men { "Alpha Manual" "Quick Start" "Alpha Commands" "Tcl Commands" \
	  "(-" "Readme" "Changes" \
	  "Extending Alpha" "Bug Reports and Debugging" "(-" }
	foreach f $men {
	    if {$f != "(-" && ![file exists [file join ${HOME} Help $f]]} {
		set men [lremove $men $f]
	    }
	}
	set ignore "" 
	foreach f [lsort $files] {
	    set f [file tail $f]
	    if {[lsearch $men $f] < 0 && [lsearch $ignore $f] < 0} {
		lappend men $f
	    }
	}
	regsub -all {\(-[ \t\r\n]+\(-} $men {\(-} men
	foreach f $men {
	    addHelpMenu $f
	}
    }
    
}

proc menu::buildwinMenu {} {
    global winMenu winNameToNum
    set ma {
	"//<Szoom"
	"//<S<I<OdefaultSize"
	"<S/;chooseAWindow"
	"/I<Biconify"
	{Menu -n arrange -p menu::winTileProc {
	    "/Jvertically^1"
	    "/J<O<Ihorizontally^2"
	    "/J<B<OunequalVert^6"
	    "/J<B<I<OunequalHor^5"
	    "(-"
	    {Menu -n other {
		{bufferOtherWindow}
		{iconify}
		{nextWin}
		{nextWindow}
		{prevWindow}
		{shrinkFull}
		{shrinkHigh}
		{shrinkLeft}
		{shrinkLow}
		{shrinkRight}
		{defaultSize}
		{swapWithNext}
		{zoom}
	    }}}
	}
	"(-"
	"/msplitWindow"
	"/otoggleScrollbar"
	"(-"
    }
    # We may be reloading, so add whatever windows we have
    if {[info exists winNameToNum]} {
	set nms [array names winNameToNum]
	foreach name $nms {
	    set item [file tail $name]
	    set num $winNameToNum($name)
	    if {$num < 10} 	{
		lappend ma /$num${item}
	    } else {
		lappend ma ${item}
	    }
	}
    }
    return [list "build" $ma menu::winProc "" $winMenu]
}

proc global::listAllBindings {} {
    new -n {* All Key Bindings *} -m Tcl -info [bindingList]
}

proc global::listGlobalBindings {} {
    global mode::features
    set text ""
    set tmp [lsort -ignore [array names mode::features]]
    foreach b [split [bindingList] "\r"] {
	set lst [lindex [split $b  " "] end]
	if {[lsearch $tmp $lst] < 0} {
	    append text "$b\r"
	}
    }
    new -n {* Global Key Bindings *} -m Tcl -info $text
}

proc global::listPackages {} {
    global index::feature
    cache::readContents index::maintainer
    foreach i [array names index::maintainer] {
	set j [lindex [set index::maintainer($i)] 1]
	set au($i) "[lindex $j 0], [lindex $j 1]"
    }
    new -n {* Installed Packages *} -m Text
    append t "Currently installed packages\r\r"
    append t "columns are: name, version, and maintainer\r"
    append t "\r\rMenus:"
    insertText $t ; set t ""
    foreach p [lsort -ignore [array names index::feature]] {
	set v [alpha::package versions $p]
	if {[lindex $v 0] == "mode"} {
	    set v "for [lindex $v 1] mode"
	}
	switch -- [lindex [set index::feature($p)] 2] {
	    "1" {
		append tm "\r[format {  %-25s %-10s  } $p $v]"
		if {[info exists au($p)]} {append tm $au($p)}
	    }
	    "0" {
		append tp "\r[format {%s %-25s %-10s  } [package::active $p { { }}] $p $v]"
		if {[info exists au($p)]} {append tp $au($p)}
	    }
	    "-1" {
		append ta "\r[format {  %-25s %-10s  } $p $v]"
		if {[info exists au($p)]} {append ta $au($p)}
	    }
	}
    }
    if {[info exists tm]} {insertText $tm ; unset tm}
    insertText "\r\rFeatures ('' = active):"
    if {[info exists tp]} {insertText $tp ; unset tp}
    insertText "\r\rAuto-loading features:"
    if {[info exists ta]} {insertText $ta ; unset ta}
    append t "\r\rModes:"
    insertText $t ; set t ""
    foreach p [lsort -ignore [alpha::package names -mode]] {
	set v [alpha::package versions $p]
	if {[lindex $v 0] == "mode"} {
	    set v "for [lindex $v 1] mode"
	}
	append t "\r[format {  %-8s %-10s  } $p $v]"
	if {[info exists au($p)]} {append t $au($p)}
    }
    insertText $t ; set t ""
    winReadOnly
    shrinkWindow
}

proc global::listFunctions {} {
    global win::Modes
    new -n {* Functions *} -m Tcl -info \
      "===\r\tCommand-double-click on a function to see its definition\r===\r\r[join [lsort -ignore [info commands]] \r]\r"
}

proc global::menusAndFeatures {} {
    global global::features mode::features mode
	
    set newGlobals [dialog::pickMenusAndFeatures global]
    set offon [package::onOrOff $newGlobals $mode 1]

    set global::features $newGlobals
    # remove removed menus
    foreach m [lindex $offon 0] {
	package::deactivate $m
    }
    foreach m [lindex $offon 1] {
	package::activate $m
    }
}

proc global::insertAllMenus {} {
    global global::features index::feature
#    foreach m ${global::features} {
#	if {[lindex [set index::feature($m)] 2] == 0} {
#	    package::activate $m
#	}
#    }
    foreach m ${global::features} {
	if {[lindex [set index::feature($m)] 2] == 1} {
	    package::activate $m
	}
    }
}

proc global::rebuildPackageIndices {} {
    if {[dialog::yesno "You must quit Alpha immediately after rebuilding. \
      Proceed?"]} {
	alpha::rebuildPackageIndices
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "menu::buildProc" --
 # 
 #  Register a procedure to be the 'build proc' for a given menu.  This
 #  procedure can do one of two things:
 #  
 #  i) build the entire menu, including evaluating the 'menu ...' command.
 #  In this case the build proc should return anything which doesn't
 #  begin 'build ...'
 #  
 #  ii) build up part of the menu, and then allow pre-registered menu
 #  insertions/replacements to take-effect.  In this case the procedure
 #  should return a list of the items (listed by index):
 #  
 #  0: "build"
 #  1: list-of-items-in-the-menu
 #  2: list of other flags.  If the list doesn't contain '-p', we use
 #  the standard menu::generalProc procedure.  If it does contain '-p'
 #  general prmenu procedure to call when an item is selected.  
 #  If nothing is given,
 #  or if '-1' is given, then we don't have a procedure.  If "" is given,
 #  we use the standard 'menu::generalProc' procedure.  Else we use the
 #  given procedure.
 #  3: list of submenus which need building.
 #  4: over-ride for the name of the menu.
 #  
 #  You must register the build-proc before attempting to build the menu.
 #  Once registered, any call of 'menu::buildSome name' will build your
 #  menu.
 # -------------------------------------------------------------------------
 ##
proc menu::buildProc {name proc} {
    global menu::build_procs
    set menu::build_procs($name) $proc
}

## 
 # -------------------------------------------------------------------------
 # 
 # "menu::insert" --
 # 
 #  name, type, where, then list of items.  type = 'items' 'submenu'
 #  
 #  Add given items to a given menu, provided they are not already there.
 #  Rebuild that menu if necessary.
 #  
 #  There are also procs 'menu::removeFrom' which does the opposite of
 #  this one, and 'menu::replaceWith' which replaces a given menu item
 #  with others.
 # -------------------------------------------------------------------------
 ##
proc menu::insert {name args} {
    if {[llength $args] < 3} { error "Too few args to menu::insert" }
    global menu::additions alpha::noMenusYet
    if {[info exists menu::additions($name)]} {
	set a [set menu::additions($name)]
	if {[lsearch -exact $a $args] != -1} { 
	    return 
	}
	# check if it's there but in a different place; we over-ride
	set dblchk [lreplace $args 1 1 "*"]
	if {[set i [lsearch -glob $a $dblchk]] == -1} {
	    unset i
	}
    }
    if {[info exists i]} {
	set menu::additions($name) [lreplace $a $i $i $args]
    } else {
	lappend menu::additions($name) $args
    }
    if {![info exists alpha::noMenusYet]} {
	# we were called after start-up; build the menu now
	menu::buildSome $name
    }
}

proc menu::uninsert {name args} {
    global menu::additions alpha::noMenusYet
    set a [set menu::additions($name)]
    if {[set idx [lsearch -exact $a $args]] == -1} { 
	return 
    }
    set menu::additions($name) [lreplace $a $idx $idx]
    if {![info exists alpha::noMenusYet]} {
	# we were called after start-up; build the menu now
	menu::buildSome $name
    }
}

proc alpha::buildMainMenus {} {
    menu::buildProc internetUpdates package::makeUpdateMenu
    menu::buildProc packages menu::packagesBuild
    menu::buildProc mode menu::modeBuild
    menu::buildProc winMenu menu::buildwinMenu
    menu::buildProc preferences menu::preferencesBuild
    uplevel #0 {
	source [file join $HOME Tcl SystemCode alphaMenus.tcl]
	menu::buildSome "File" "Edit" "Text" "Search" "Utils" "Config" "winMenu"
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "menu::buildSome" --
 # 
 #  Important procedure which builds all known/registered menus from a
 #  number of pieces.  It allows the inclusion of menus pieces registered
 #  with the menu::insert procedure, which allows you easily to add items
 #  (including dynamic and hierarchial) to any of Alpha's menus.
 # 
 # Results:
 #  Various menus are (re)built
 # 
 # Side effects:
 #  Items added to those menus with 'addMenuItem' will vanish.
 # 
 # --Version--Author------------------Changes-------------------------------
 #    1.0     <vince@santafe.edu> original
 #    2.0     <vince@santafe.edu> more compact, more like tk
 # -------------------------------------------------------------------------
 ##
proc menu::buildSome {args} {
    set msubs {}
    foreach token $args {
	eval lappend msubs [menu::buildOne $token]
    }
    # build sub-menus of those built
    if {[llength $msubs]} {eval menu::buildSome $msubs}
}

proc menu::buildOne {args} {
    global menu::additions menu::build_procs alpha::noMenusYet \
      menu::items
    set token [lindex $args 0] ; set args [lrange $args 1 end]
    if {[set len [llength $args]] > 0 || [info exists menu::build_procs($token)]} {
	if {$len > 0} {
	    set res $args
	} else {
	    if {[catch "[set menu::build_procs($token)]" res]} {
		alpha::reportError "The menu $token had a problem starting up ; $res"
	    }
	}
	switch -- [lindex $res 0] {
	    "build" {
		set ma [lindex $res 1]
		if {[llength $res] > 2} {
		    set theotherflags [lrange [lindex $res 2] 1 end]
		    if {[lindex [lindex $res 2] 0] != -1} {
			set mproc [lindex [lindex $res 2] 0]
		    }
		    if {[lindex $res 3] != ""} {
			eval lappend msubs [lindex $res 3]
		    }
		    if {[lindex $res 4] != ""} { set name [lindex $res 4] }
		}
	    } "menu" - "Menu" {
		eval $res
		return ""
	    } default {
		return ""
	    }
	}
    } else {
	set ma ""
	if {[info exists menu::items($token)]} {
	    set ma [set menu::items($token)]
	    global menu::proc menu::which_subs menu::otherflags
	    if {[info exists menu::proc($token)]} {
		set mproc [set menu::proc($token)]
	    }
	    if {[info exists menu::which_subs($token)]} {
		eval lappend msubs [set menu::which_subs($token)]
	    }
	    if {[info exists menu::otherflags($token)]} {
		set theotherflags [set menu::otherflags($token)]
	    }
	}
    }

    if {![info exists name]} { set name $token }
    # add any registered items and make the menu contents
    if {[info exists menu::additions($token)]} {
	foreach ins [set menu::additions($token)] {
	    set where [lindex $ins 1]
	    set type [lindex $ins 0]
	    set ins [lrange $ins 2 end]
	    switch -- $type {
		"submenu" {
		    lappend msubs [lindex $ins 0]
		    set ins [list [list Menu -n [lindex $ins 0] {}]]
		}
	    }
	    switch -- [lindex $where 0] {
		"replace" {
		    set old [lindex $where 1]
		    if {[set ix [eval llindex ma $old]] != -1} {
			set ma [eval [list lreplace $ma $ix [expr {$ix -1 + [llength $old]}]] $ins]
		    } else {
			alertnote "Bad menu::replacement registered '$old'"
		    }
		    
		}
		"end" {
		    eval lappend ma $ins
		}
		default {
		    set ma [eval linsert [list $ma] $where $ins]
		}
	    }
	}
    }
    # These two lines removed due to some conflicts
    #    regsub -all {"?\(-"?([ \t\r\n]+"?\(-"?)+} $ma "(-" ma
    #    regsub -all {(^[ \t\r\n]*"?\(-"?|"?\(-"?[ \t\r\n]*$)} $ma "" ma

    # backwards compatibility fix.  Removed because it's inefficient,
    # and it's about time people used the new Menu command ;-)
    # regsub -all "\{menu " $ma "\{Menu " ma

    # build the menu
    set name [list -n $name]
    if {[info exists theotherflags]} {
	set name [concat $theotherflags $name]
    }
    if {[info tclversion] >= 8.0} {
	lappend name -h [list "This is the [lindex $name end] menu"]
    }
    if {[info exists mproc]} {
	if {$mproc != ""} {
	    eval Menu $name -p $mproc [list $ma]
	} else {
	    eval Menu $name [list $ma]
	}
    } else {
	eval Menu $name -p menu::generalProc [list $ma]
    }
    if {[info exists msubs]} {
	return $msubs
    }
    return ""
}

proc menu::replaceRebuild {name title} {
    global $name
    catch {removeMenu [set $name]}
    set $name $title
    menu::buildSome $name
    insertMenu [set $name]
}

proc menu::packagesBuild {} {
    global alpha::package_menus package::prefs
    lappend ma [menu::itemWithIcon "packagePreferences" 84] \
      "miscellaneousPackages"
    if {[info exists package::prefs]} {
	foreach pkg ${package::prefs} {
	    lappend ma "${pkg}Prefs"
	}
    }
    lappend ma "(-" "describeAPackage" "readHelpForAPackage" \
      "uninstallSomePackages" \
      {Menu -m -n internetUpdates -p package::menuProc {}} \
      "(-" "rebuildPackageIndices"
    return [list build $ma menu::packagesProc internetUpdates]
}

proc menu::packagesProc {menu item} {
    global package::prefs
    if {[regexp "(.*)Prefs" $item d pkg]} {
	if {[lcontains package::prefs $pkg]} {
	    dialog::pkg_options $pkg
	    return
	}
    }
    switch -- $item {
	"miscellaneousPackages" {
	    return [dialog::preferences $menu Packages]
	}
	"describeAPackage" -
	"Describe A Package" {
	    set pkg [dialog::optionMenu "Describe which package?" \
	      [lsort -ignore [alpha::package names]]]
	    package::describe $pkg
	}
	"readHelpForAPackage" -
	"Read Help For A Package" {
	    set pkg [dialog::optionMenu "Read help for which package?" \
	      [lsort -ignore [alpha::package names]]]
	    package::helpFile $pkg
	}
	"uninstallSomePackages" -
	"Uninstall Some Packages" {
	    package::uninstall
	}
	"rebuildPackageIndex" {
	    alpha::rebuildPackageIndices
	}
	"packagePreferences" {
	    alertnote "Select a package from the group below in the menu to\
	      edit its preferences."
	}
	default {
	    menu::generalProc global $item
	}
    }
}


proc menu::menuPackages {menu m} {
    if {[package::helpOrDescribe $m]} {
	return
    }
    # toggle global existence of '$m' menu
    global global::menus modifiedVars
    if {[set idx [lsearch  ${global::menus} $m]] == -1} {
	lappend global::menus $m
	global $m
	catch $m
	insertMenu [set	$m]
	markMenuItem packageMenus $m 1
    } else {
	set global::menus [lreplace ${global::menus} $idx $idx]
	global $m
	catch "removeMenu [set $m]"
	markMenuItem packageMenus $m 0
    }
    lappend modifiedVars global::menus
}

if {[info tclversion] < 8.0} {
    proc menu::modeBuild {} {
	set ma [list "/p<BmenusAndFeatures" "/ppreferences" "editPrefsFile" \
	  "loadPrefsFile" "describeMode" "(-" "/m<UchangeMode"]
	return [list build $ma mode::menuProc "" "Mode Prefs"]
    }
} else {
    proc menu::modeBuild {} {
	global mode
	set ma [list "/p<BmenusAndFeatures" "/ppreferences" "editPrefsFile" \
	  "loadPrefsFile" "describeMode" "(-" "/m<UchangeMode"]
	if {$mode != ""} {
	    return [list build $ma mode::menuProc "" "${mode} Mode Prefs"]
	} else {
	    return [list build $ma mode::menuProc "" "Mode Prefs"]
	}
    }
}

proc menu::preferencesBuild {} {
    global flagPrefs
    
    set ma [list "/p<U<BMenus And Features" "/p<USuffix Mappings" \
      "Edit Prefs File" "(-" [menu::itemWithIcon "Interface Preferences" 84]]
    lappend ma Appearance Electrics Text Tiling Window "(-" \
      [menu::itemWithIcon "Input-Output Preferences" 84]
    lappend ma Backups Files Printer Tags WWW "(-" \
      [menu::itemWithIcon "System Preferences" 84]
    eval lunion ma [lsort [lremove [array names flagPrefs] Packages]]
    return [list build $ma {dialog::preferences -m}]
}

proc menu::removeFrom {name args} {
    global menu::additions alpha::noMenusYet
    if {[info exists menu::additions($name)]} {
	if {[set i [lsearch -exact [set menu::additions($name)] $args]] != -1} {
	    set menu::additions($name) [lreplace [set menu::additions($name)] $i $i]
	    if {![info exists alpha::noMenusYet]} {
		# we were called after start-up; build the menu now
		menu::buildSome $name
	    }
	}
    }
}

proc menu::replaceWith {name current type args} {
    global menu::additions alpha::noMenusYet
    if {![info exists menu::additions($name)]} {
	lappend menu::additions($name) [concat [list $type [list replace $current]] $args]
    } else {
	set add 1
	set j 0
	foreach i [set menu::additions($name)] {
	    if {[lrange $i 0 1] == [list $type [list replace $current]]} {
		if {[lindex $i 1] != $args} {
		    set add 0
		    set menu::additions($name) \
		      [lreplace [set menu::additions($name)] $j $j \
		      [concat [list $type [list replace $current]] $args]]
		    break
		} else {
		    # no change
		    return
		}
	    }
	    incr j
	}
	if {$add} {
	    lappend menu::additions($name) [concat [list $type [list replace $current]] $args]
	}
    }
    if {![info exists alpha::noMenusYet]} {
	# we were called after start-up; build the menu now
	menu::buildSome $name
    }
}

proc menu::itemWithIcon {name icon} {
    return "/\x1e${name}^[text::Ascii $icon 1]"
}

proc menu::fileProc {menu item} {
    switch -- $item {
	"open" {
	    findFile
	}
	"close" {
	    killWindow
	}
	default {
	    uplevel 1 [list menu::generalProc file $item]
	}
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "menu::generalProc" --
 # 
 #  If either 'item' or 'menu::item' exists, call it.  Else try and
 #  autoload 'item', if that fails try and autoload 'menu::item'
 # -------------------------------------------------------------------------
 ##
if {[info tclversion] < 8.0} {
    proc menu::generalProc {menu item {lower 1}} {
	if {$lower} {set menu [string tolower $menu]}
	if {[info commands ${menu}::${item}] != ""} {
	    uplevel \#0 ${menu}::$item
	} elseif {[info commands $item] != ""} {
	    uplevel \#0 $item
	} elseif {[auto_load ${menu}::$item]} {
	    uplevel \#0 ${menu}::$item
	} else {
	    uplevel \#0 $item
	}
    }
} else {
    proc menu::generalProc {menu item {lower 1}} {
	if {$lower} {set menu [string tolower $menu]}
	if {[info commands ::${menu}::${item}] != ""} {
	    uplevel \#0 ::${menu}::$item
	} elseif {[info commands $item] != ""} {
	    uplevel \#0 $item
	} elseif {[auto_load ::${menu}::$item]} {
	    uplevel \#0 ::${menu}::$item
	} else {
	    uplevel \#0 $item
	}
    }
}

proc menu::globalProc {menu item} {
    menu::generalProc global $item
}

proc menu::winProc {menu name} {
    global winNameToNum

    set nms [array names winNameToNum]

    if {[lsearch -glob $nms "*[quote::Find $name]"] < 0} {
        $name
        return
    }

    foreach nm $nms {
        if {[string match *[quote::Find $name] $nm] == "1"}  {
            bringToFront $name
            if {[icon -q]} { icon -f $name -o }
            return
        }
    }
    return "normal"
}


## 
 # proc namedClipMenuProc {menu item} {
 #     switch $item {
 #         "copy"      "copyNamedClipboard"
 #         "cut"       "cutNamedClipboard"
 #         "paste"     "pasteNamedClipboard"
 #     }
 # }
 ##

proc menu::colorProc {menu item} {
    global colorInds modifiedArrVars
    if {[info exists colorInds($item)]} {
	set color [eval [list colorTriple "New \"$item\":"] $colorInds($item)]
    } else {
	switch -- $item {
	    foreground	{ set inds "0 0 0" }
	    background	{ set inds "65535 65535 65535" }
	    blue		{ set inds "0 0 65535" }
	    cyan		{ set inds "61404 11464 34250" }
	    green		{ set inds "1151 33551 8297" }
	    magenta		{ set inds "44790 1591 51333" }
	    red			{ set inds "65535 0 0" }
	    white		{ set inds "65535 65535 65535" }
	    yellow		{ set inds "61834 64156 12512" }
	    default		{ set inds "65535 65535 65535" }
	}
	set color [eval [list colorTriple "New \"$item\":"] $inds]
    }
    eval setRGB $item $color
    
    set colorInds($item) $color
    alpha::makeColourList
    lappend modifiedArrVars colorInds
}

proc alpha::makeColourList {} {
    global alpha::colors colorInds alpha::basiccolors
    # Set up color indices
    foreach ind [array names colorInds] {
	eval setRGB $ind $colorInds($ind)
    }
    set alpha::basiccolors {none blue cyan green magenta red white yellow}
    set alpha::colors ${alpha::basiccolors}
    foreach c {color_9 color_10 color_11 color_12 color_13 color_14 color_15} {
	if {[info exists colorInds($c)]} {lappend alpha::colors $c}
    }
}


		
#===============================================================================
proc helpMenu {item} {
    global HOME
    edit -r -c [file join $HOME Help $item]
}

## 
 # -------------------------------------------------------------------------
 # 
 # "alphaHelp" --
 # 
 #  Called from about box
 # -------------------------------------------------------------------------
 ##
proc alphaHelp {} {
    global HOME
    if {[file exists [set f [file join ${HOME} Help "Alpha Manual"]]]} {
	edit -r -c $f
    } else {
	edit -r -c [file join $HOME Help "Quick Start"]
    }
}

proc register {} {
    global HOME
    launch -f [file join $HOME Register]
}

namespace eval icon {}
namespace eval file {}

proc icon::FromSig {sig} {
    global alpha::_icons
    if {[set p [lsearch -glob ${alpha::_icons} "[quote::Find ${sig}] *"]] != -1} {
	set p [lindex ${alpha::_icons} $p]
	return [lindex $p 2]
    } else {
	return ""
    }
}

proc icon::MenuFromSig {sig} {
    global alpha::_icons
    if {[set p [lsearch -glob ${alpha::_icons} "[quote::Find ${sig}] *"]] != -1} {
	set char [expr {[lindex [lindex ${alpha::_icons} $p] 2] -208}]
	if {$char < 1 || $char > 256} { return "" }
	return "^[text::Ascii $char 1]"
    } else {
	return ""
    }
}


proc menu::fileUtils {menu item} {
    if {[lsearch -exact {"insertPathName" "insertFile" "fileRemove" "fileInfo" "wordCount" "textToAlpha"} $item] != -1} {return [$item]}
    switch -- $menu {
	"moreUtils" {
	    file::Utils::$item
	}
	default {
	    file::$item
	}
    }
}

proc menu::winTileProc {menu item} {
    win$item
}

## 
 # -------------------------------------------------------------------------
 # 
 #	"menu::buildHierarchy" --
 # 
 #  Given a list of folders, 'menu::buildHierarchy' returns a hierarchical
 #  menu based on the files and subfolders in each of these folders. 
 #  Pathnames are optionally stored in a global array given by the argument
 #  'filePaths'.  The path's index in this array is formed by concatenating
 #  the submenu name and the filename, allowing the pathname to be
 #  retrieved by the procedure 'proc' when the menu item is selected.
 # 
 #  The search may be restricted to files with specific extensions, or
 #  files matching a certain pattern.  A search depth may also be given,
 #  with three levels of subfolders assumed by default.
 # 
 #  See MacPerl.tcl or latexMenu.tcl for examples.
 # 
 #  (originally written by Tom Pollard, with modifications by Vince Darley
 #  and Tom Scavo)
 # 
 # --Version--Author------------------Changes-------------------------------
 #	  1.0	  Tom Pollard			        original
 #	  2.0	  <vince@das.harvard.edu> multiple extensions, optional	paths
 #	  2.1	  Tom Scavo				        multiple folders
 #	  2.2	  <vince@das.harvard.edu> pattern matching as well as exts
 #	  2.3	  <vince@das.harvard.edu> handles unique menu-names and does text only
 #	  2.4	  <jl@theophys.kth.se>    now also handles patterns like "*.{a,b}"
 # -------------------------------------------------------------------------
 ##
proc menu::buildHierarchy {folders name proc {filePaths {}} {exts *} {depth 3} {fset {}}} {
    global filesetmodeVars file::separator
    if { $filePaths != "" } {
	global $filePaths
    }
    if {[llength $exts] > 1} {
	regsub -all {\.} $exts "" exts
	set exts "*.{[join $exts ,]}"
    } elseif {[string match ".*" $exts] && ![string match {*\**} $exts]} {set exts "*$exts"}
    incr depth -1
    set overallMenu {}
    foreach folder $folders {
	if {[file exists $folder]} {
	    if {![file isdirectory $folder]} {
		set folder "[file dirname $folder]${file::separator}"
	    }
	    if {![regexp -- "${file::separator}$" $folder]} {
		set folder "$folder${file::separator}"
	    }
	    if {$name == 0} {
		set name [file tail [file dirname ${folder}dummy]]
	    }
	    # if it's a fileset, we register _before_ recursing
	    if { $fset != "" } {
		set mname [registerFilesetMenuName $fset $name $proc]
	    } else {
		set mname $name
	    }
	    set menu {}
	    set subfolders [glob -nocomplain -t d -path $folder *]
	    if {$filesetmodeVars(includeNonTextFiles)} {
		set filenames [glob -nocomplain -path $folder -- $exts]
	    } else {
		set filenames [glob -t TEXT -nocomplain -path $folder $exts]
	    }
	    # Note that the list of filenames may also contain some/all
	    # subfolders (if they matched the glob expression), hence
	    # we must be sure not to add them twice.
	    foreach m [lsort -ignore [concat $subfolders $filenames]] {
		if {[set s [lsearch -exact $subfolders $m]] != -1 && $depth > 0} {
		    set subM [menu::buildHierarchy [list ${m}] 0 $proc $filePaths $exts $depth $fset]
		    if {[llength $subM]} { lappend menu $subM }
		} elseif {[file isfile $m]} {
		    lappend menu [set fname [file tail $m]]
		    if { $filePaths != "" } {
			set ${filePaths}([file join $name $fname]) $m
		    }
		}
	    }
	    
	    if {[llength $menu]} {
		set overallMenu [concat $overallMenu $menu]
	    }
	} else {
	    beep
	    alertnote "menu::buildHierarchy:  Folder $folder does not exist!"
	}
    }
    
    if {[llength $overallMenu]} {
	if { [string length $proc] > 1 } {
	    set pproc "-p $proc"
	} else {
	    set pproc ""
	}	
	if { $fset != "" } {
	    if { [string length $proc] > 1 } { set pproc "-p subMenuProc" }
	}	 
	return [concat {Menu -m -n} [list $mname] $pproc [list $overallMenu]]
	
    } else {
	return ""
    }
}

# in case we've done something odd elsewhere
ensureset filesetmodeVars(includeNonTextFiles) 0


proc menu::reinterpretOldMenu {args} {
    set ma [lindex $args end]
    set args [lreplace $args end end]
    getOpts {-n -M -p}
    if {[info exists opts(-p)]} {
	lappend proc $opts(-p)
    } else {
	lappend proc "-1"
    }
    if {[info exists opts(-M)]} { lappend proc -M $opts(-m) }
    if {[info exists opts(-m)]} { lappend proc -m }
    menu::buildOne $opts(-n) build $ma $proc
}




