/*************************************\
 *                                   *
 *       P R I N T    F I N E        *
 *                                   *
 *  ASCII to TeX conversion system   *
 *                                   *
 * Copyright (C) 1994 Klaus Bechert  *
 *                                   *
\*************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


/*
 * function prototypes
 */
extern int cfg(char *name);


/*
 * directory separator
 */
#define DS '/'
#ifdef __TOS__
#undef DS
#define DS '\\'
#endif
#ifdef __MSDOS__
#undef DS
#define DS '\\'
#endif


/*
 * globals
 */
#define LINESIZE 2047

char buf[128];             /* buffer for filenames */
FILE *sf, *df;             /* file descriptors for source and destination */
char linebuf[LINESIZE+1];  /* max. character per line */
int  verbose;              /* normal=0, verbose=1 */
int  formfeed;             /* normal=1, ignore=0 */

/*
 * externals
 */
extern char *fontname;
extern int columns;
extern int tabstop;
extern int linesize;
extern double hsize, vsize, hoffset, voffset;
extern double widthcolumn;

/*
 * print help text and terminate
 */
void help(void)
{
 printf("usage:\n");
 printf("pf [-options] inputfile [configurationfile]\n\n");
 printf("valid options:\n");
 printf("  -oXXX    output file name = XXX\n");
 printf("  -v       verbose output\n");
 printf("  -i       ignore form feeds\n");
 exit(1);
}


/*
 * the program
 */
main(int argc, char *argv[])
{
 char *p, *q, *ifi, *cfi, *ofi;
 int n, c, col, tab;
 long toolong, line;

 /* no arguments: help! */
 if (argc < 2) help();

 /* look for options and filenames */
 for (formfeed=1, verbose=0, ifi=cfi=ofi=NULL, n=1; n<argc; n++)
 {
  if (argv[n][0] == '-')
  {
   switch(argv[n][1])
   {
    case 'o': case 'O':
     ofi = argv[n]+2;
     break;
    case 'v': case 'V':
     verbose = 1;
     printf("print fine, an ASCII to TeX converter, Version 1.0\n");
     printf("Copyright (C) 6/1994, Klaus Bechert\n\n");
     break;
    case 'i': case 'I':
     formfeed = 0;
     break;
    case '?':  /* for future use !! */
     ;
   }
  }
  else
  {
   if (ifi == NULL) ifi = argv[n];
   else if (cfi == NULL) cfi = argv[n];
   else help();
  }
 }
 if (ifi == NULL) help();

 /* look for configuration file */
 if (cfi)
 {
  p = getenv("PRINTFINE");
  if (p==NULL)
  {
   strcpy(buf, argv[0]);
   if (buf[0])
   {
    p = strrchr(buf, DS);
    if (p) *(++p)=0; else buf[0]=0;
   }
   else
   {
    buf[0]=0;
   }
  }
  else
  {
   strcpy(buf, p);
   p = strchr(buf, 0);
   *(p++) = DS; *p = 0;
  }
  strcat(buf, cfi);
  if ((p = strrchr(buf, DS)) == NULL) p = buf; else p++;
  if ((q = strrchr(p, '.')) == NULL) strcat(buf, ".cfg");
  if (cfg(buf) < 0) return 2;
 }

 /* test configuration */
 if (columns > 3)
 {
  fprintf(stderr, "the value of 'columns' must not be > 3\n");
  exit(2);
 }
 if (linesize > LINESIZE)
 {
  fprintf(stderr, "the value of 'linesize' must not be > %d\n", LINESIZE);
  exit(2);
 }

 /* make an output file name if no o-option given */
 if (ofi == NULL)
 {
  strcpy(buf, ifi);
  if ((p = strchr(buf, ':')) == NULL) p = buf; else p++;
  if ((q = strrchr(p, DS)) == NULL) q = p; else q++;
  if ((p = strchr(q, '.')) == NULL) p = strchr(q, 0);
  *p = 0; strcat(buf, ".tex");
  ofi = buf;
 }

 /* test input file name against output file name */
 if (strcmp(ifi, ofi) == 0)
 {
  fprintf(stderr, "*** Attention! input and output file "
                  "have the same name ***\n");
  return 1;
 }

 /* verbosity */
 if (verbose)
 {
  printf("input  file   = %s\n", ifi);
  printf("output file   = %s\n", ofi);
  printf("TeX font name = %s\n", fontname);
  p = columns == 1 ? "" : "s";
  printf("Output %d column%s, max. %d characters per line\n\n",
          columns, p, linesize);
 }

 /* open input file */
 sf = fopen(ifi, "r");
 if (sf == NULL)
 {
  fprintf(stderr, "cannot open %s\n", ifi);
  exit(2);
 }

 /* input file name without its path */
 if ((p = strrchr(ifi, DS)) != NULL) ifi = p+1;

 /* open output file */
 df = fopen(ofi, "w");
 if (df == NULL)
 {
  fprintf(stderr, "cannot open %s\n", ofi);
  exit(2);
 }

 /* print banner */
 fprintf(df, "%% ASCII->TeX conversion, Copyright (C) 6/1994,"
             " Klaus Bechert\n");
 fprintf(df, "%% filename = %s\n", ifi);

 /* how many columns */
 if (columns == 1)
 {
  fprintf(df, "%% one column output\n");
  fprintf(df, "%% max. %d characters per line\n%%\n", linesize);
  fprintf(df, "\\hsize=%lgin\n", hsize);
  fprintf(df, "\\def\\onecol{\\shipout\n");
  fprintf(df, "\\vbox{\\hbox to \\hsize{\\rm\\filename\n");
  fprintf(df, "\\hfill page \\folio}\\smallskip\\hrule\n");
  fprintf(df, "\\hbox to \\hsize{\\leftline{\\pagebody}}\n");
  fprintf(df, "}\\advancepageno\n");
  fprintf(df, "\\ifnum\\outputpenalty>-20000\\else\\dosupereject\\fi}\n");
  fprintf(df, "\\output={\\onecol}\n");
 }
 else if (columns == 2)
 {
  fprintf(df, "%% two columns output\n");
  fprintf(df, "%% max. %d characters per line\n%%\n", linesize);
  fprintf(df, "\\newdimen\\Hsize\\newbox\\firstcol\n");
  fprintf(df, "\\newif\\ifsecondcol\\secondcolfalse\n");
  fprintf(df, "\\Hsize=%lgin\\hsize=%lgin\n", hsize, widthcolumn);
  fprintf(df, "\\def\\twocol{\\ifsecondcol\\shipout\n");
  fprintf(df, "\\vbox{\\hbox to \\Hsize{\\rm\\filename\n");
  fprintf(df, "\\hfill page \\folio}\\smallskip\\hrule\n");
  fprintf(df, "\\hbox to \\Hsize{\\leftline{\\box\\firstcol}\n");
  fprintf(df, "\\vrule\\hfill\\leftline{\\pagebody}}%%\\longfootline\n");
  fprintf(df, "}\\global\\secondcolfalse\\advancepageno\\else\n");
  fprintf(df, "\\global\\setbox\\firstcol\\pagebody\n");
  fprintf(df, "\\global\\secondcoltrue\\fi\n");
  fprintf(df, "\\ifnum\\outputpenalty>-20000\n");
  fprintf(df, "\\else\\dosupereject\\fi}\n");
  fprintf(df, "\\output={\\twocol}\n");
 }
 else if (columns == 3)
 {
  fprintf(df, "%% three columns output\n");
  fprintf(df, "%% max. %d characters per line\n%%\n", linesize);
  fprintf(df, "\\newdimen\\Hsize\\newbox\\firstcol\\newbox\\secondcol\n");
  fprintf(df, "\\newif\\ifsecondcol\\secondcolfalse\n");
  fprintf(df, "\\newif\\ifthirdcol\\thirdcolfalse\n");
  fprintf(df, "\\Hsize=%lgin\\hsize=%lgin\n", hsize, widthcolumn);
  fprintf(df, "\\def\\threecol{\\ifthirdcol\\shipout\n");
  fprintf(df, "\\vbox{\\hbox to \\Hsize{\\rm\\filename\n");
  fprintf(df, "\\hfill page \\folio}\\smallskip\\hrule\n");
  fprintf(df, "\\hbox to \\Hsize{\\leftline{\\box\\firstcol}\n");
  fprintf(df, "\\vrule\\hfill\\leftline{\\box\\secondcol}\n");
  fprintf(df, "\\vrule\\hfill\\leftline{\\pagebody}}%%\\longfootline\n");
  fprintf(df, "}\\advancepageno\\global\\thirdcolfalse\\else\n");
  fprintf(df, "\\ifsecondcol\\global\\setbox\\secondcol\\pagebody\n");
  fprintf(df, "\\global\\secondcolfalse\\global\\thirdcoltrue\\else\n");
  fprintf(df, "\\global\\setbox\\firstcol\\pagebody\n");
  fprintf(df, "\\global\\secondcoltrue\\fi\\fi\n");
  fprintf(df, "\\ifnum\\outputpenalty>-20000\\else\\dosupereject\\fi}\n");
  fprintf(df, "\\output={\\threecol}\n");
 }

 /* Begin of the Text */
 fprintf (df, "{\\font\\pc = %s\\pc\n", fontname);
 fprintf (df, "\\vsize=%lgin", vsize);
 fprintf (df, "\\hoffset=%lgin\\voffset=%lgin\n", hoffset, voffset);
 fprintf (df, "\\parindent=0pt\\baselineskip=\\fontdimen8\\pc\n");
 fprintf (df, "\\lineskip=0pt\n");
 fprintf (df, "\\newskip\\emptyline\\emptyline=\\the");
 fprintf (df, "\\baselineskip plus\\the\\baselineskip\n");
 fprintf (df, "\\catcode`\\#=12\\catcode`\\$=12\\catcode`\\~=12\n");
 fprintf (df, "\\catcode`\\&=12\\catcode`\\^=12\\catcode`\\_=12\n");
 fprintf (df, "\\catcode`\\ =12\n");

 /* Filename */
 fprintf (df, "\\gdef\\filename{%s}\n", ifi);
 line = toolong = 0;

 /* loop */
 do
 {
  /* read one line from source file */
  for (n=0, c=getc(sf); n<linesize && c!=EOF && c!='\n'; n++, c=getc(sf))
   linebuf[n] = c;
  line++; col=0; linebuf[n] = 0;

  /* line too long ? */
  if (c!='\n' && c!=EOF)
  {
   fprintf(stderr, "line %ld of inputfile is too "
                   "long (will be truncated)\n", line);
   toolong++;
   while (c!='\n' && c!=EOF) c=getc(sf);
  }

  /* is this line empty ? */
  for (p=linebuf; *p==' ' || *p=='\t'; p++);
  if (*p==0)
  {
   fprintf(df, "\\vskip\\emptyline\\goodbreak\n");
  }
  else
  {
   for (p=linebuf; *p; p++)
   {
    /* tab ? */
    if (*p == '\t')
    {
     tab = ((col+tabstop)/tabstop)*tabstop;
     if (tab > linesize) break;
     do {fprintf(df, " ");} while (++col < tab);
    }
    /* form feed */
    else if (*p == '\f' && formfeed)
    {
     fprintf(df, "\\vfill\\eject\n");
    }
    else
    {
     if (++col > linesize) break;
     if ((*p < ' ') || (strchr("\\{}", *p)))
        fprintf (df, "\\char%d{}", ((int)*p & 0xFF));
     else if (*p == '%')
        fprintf (df, "\\%%");
     else
        fprintf (df, "%c", *p);
    }
   }
  }
  fprintf(df, "\\par\n");
 } while (c != EOF);

 /* end of the TeX file */
 fprintf(df, "}\\bye\n");

 /* close the files */
 fclose(df);
 fclose(sf);

 /* snack */
 if (verbose)
 {
  printf("%ld lines converted, %ld too long lines truncated\n\n",
           line, toolong);
 }
 return 0;
}