/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.jdt.internal.ui.wizards.buildpaths;

import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;

import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.jface.window.Window;

import org.eclipse.ui.dialogs.ElementTreeSelectionDialog;
import org.eclipse.ui.dialogs.ISelectionStatusValidator;
import org.eclipse.ui.help.WorkbenchHelp;
import org.eclipse.ui.model.WorkbenchContentProvider;
import org.eclipse.ui.model.WorkbenchLabelProvider;
import org.eclipse.ui.views.navigator.ResourceSorter;

import org.eclipse.jdt.internal.ui.IJavaHelpContextIds;
import org.eclipse.jdt.internal.ui.dialogs.StatusDialog;
import org.eclipse.jdt.internal.ui.dialogs.StatusInfo;
import org.eclipse.jdt.internal.ui.wizards.NewWizardMessages;
import org.eclipse.jdt.internal.ui.wizards.TypedElementSelectionValidator;
import org.eclipse.jdt.internal.ui.wizards.TypedViewerFilter;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.DialogField;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.IDialogFieldListener;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.IStringButtonAdapter;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.LayoutUtil;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.StringButtonDialogField;

public class ExclusionInclusionEntryDialog extends StatusDialog {
	
	private StringButtonDialogField fExclusionPatternDialog;
	private StatusInfo fExclusionPatternStatus;
	
	private IContainer fCurrSourceFolder;
	private String fExclusionPattern;
	private List fExistingPatterns;
	private boolean fIsExclusion;
		
	public ExclusionInclusionEntryDialog(Shell parent, boolean isExclusion, String patternToEdit, List existingPatterns, CPListElement entryToEdit) {
		super(parent);
		fIsExclusion= isExclusion;
		fExistingPatterns= existingPatterns;
		String title, message;
		if (isExclusion) {
			if (patternToEdit == null) {
				title= NewWizardMessages.getString("ExclusionInclusionEntryDialog.exclude.add.title"); //$NON-NLS-1$
			} else {
				title= NewWizardMessages.getString("ExclusionInclusionEntryDialog.exclude.edit.title"); //$NON-NLS-1$
			}
			message= NewWizardMessages.getFormattedString("ExclusionInclusionEntryDialog.exclude.pattern.label", entryToEdit.getPath().makeRelative().toString());  //$NON-NLS-1$
		} else {
			if (patternToEdit == null) {
				title= NewWizardMessages.getString("ExclusionInclusionEntryDialog.include.add.title"); //$NON-NLS-1$
			} else {
				title= NewWizardMessages.getString("ExclusionInclusionEntryDialog.include.edit.title"); //$NON-NLS-1$
			}
			message= NewWizardMessages.getFormattedString("ExclusionInclusionEntryDialog.include.pattern.label", entryToEdit.getPath().makeRelative().toString());  //$NON-NLS-1$
		}
		setTitle(title);
		if (patternToEdit != null) {
			fExistingPatterns.remove(patternToEdit);
		}
		
		
		IWorkspaceRoot root= entryToEdit.getJavaProject().getProject().getWorkspace().getRoot();
		IResource res= root.findMember(entryToEdit.getPath());
		if (res instanceof IContainer) {
			fCurrSourceFolder= (IContainer) res;
		}		
		
		fExclusionPatternStatus= new StatusInfo();
		
		ExclusionPatternAdapter adapter= new ExclusionPatternAdapter();
		fExclusionPatternDialog= new StringButtonDialogField(adapter);
		fExclusionPatternDialog.setLabelText(message);
		fExclusionPatternDialog.setButtonLabel(NewWizardMessages.getString("ExclusionInclusionEntryDialog.pattern.button")); //$NON-NLS-1$
		fExclusionPatternDialog.setDialogFieldListener(adapter);
		fExclusionPatternDialog.enableButton(fCurrSourceFolder != null);
		
		if (patternToEdit == null) {
			fExclusionPatternDialog.setText(""); //$NON-NLS-1$
		} else {
			fExclusionPatternDialog.setText(patternToEdit.toString());
		}
	}
	
	
	protected Control createDialogArea(Composite parent) {
		Composite composite= (Composite)super.createDialogArea(parent);
		
		int widthHint= convertWidthInCharsToPixels(60);
		
		Composite inner= new Composite(composite, SWT.NONE);
		GridLayout layout= new GridLayout();
		layout.marginHeight= 0;
		layout.marginWidth= 0;
		layout.numColumns= 2;
		inner.setLayout(layout);
		
		Label description= new Label(inner, SWT.WRAP);
		
		if (fIsExclusion) {
			description.setText(NewWizardMessages.getString("ExclusionInclusionEntryDialog.exclude.description")); //$NON-NLS-1$
		} else {
			description.setText(NewWizardMessages.getString("ExclusionInclusionEntryDialog.include.description")); //$NON-NLS-1$
		}
		GridData gd= new GridData();
		gd.horizontalSpan= 2;
		gd.widthHint= convertWidthInCharsToPixels(80);
		description.setLayoutData(gd);
		
		fExclusionPatternDialog.doFillIntoGrid(inner, 3);
		
		LayoutUtil.setWidthHint(fExclusionPatternDialog.getLabelControl(null), widthHint);
		LayoutUtil.setHorizontalSpan(fExclusionPatternDialog.getLabelControl(null), 2);
		
		LayoutUtil.setWidthHint(fExclusionPatternDialog.getTextControl(null), widthHint);
		LayoutUtil.setHorizontalGrabbing(fExclusionPatternDialog.getTextControl(null));
				
		fExclusionPatternDialog.postSetFocusOnDialogField(parent.getDisplay());
		applyDialogFont(composite);		
		return composite;
	}

		
	// -------- ExclusionPatternAdapter --------

	private class ExclusionPatternAdapter implements IDialogFieldListener, IStringButtonAdapter {
		
		// -------- IDialogFieldListener
		
		public void dialogFieldChanged(DialogField field) {
			doStatusLineUpdate();
		}

		public void changeControlPressed(DialogField field) {
			doChangeControlPressed();
		}
	}
	
	protected void doChangeControlPressed() {
		IPath pattern= chooseExclusionPattern();
		if (pattern != null) {
			fExclusionPatternDialog.setText(pattern.toString());
		}
	}

	protected void doStatusLineUpdate() {
		checkIfPatternValid();
		updateStatus(fExclusionPatternStatus);
	}		
	
	protected void checkIfPatternValid() {
		String pattern= fExclusionPatternDialog.getText().trim();
		if (pattern.length() == 0) {
			fExclusionPatternStatus.setError(NewWizardMessages.getString("ExclusionInclusionEntryDialog.error.empty")); //$NON-NLS-1$
			return;
		}
		IPath path= new Path(pattern);
		if (path.isAbsolute() || path.getDevice() != null) {
			fExclusionPatternStatus.setError(NewWizardMessages.getString("ExclusionInclusionEntryDialog.error.notrelative")); //$NON-NLS-1$
			return;
		}
		if (fExistingPatterns.contains(pattern)) {
			fExclusionPatternStatus.setError(NewWizardMessages.getString("ExclusionInclusionEntryDialog.error.exists")); //$NON-NLS-1$
			return;
		}
		
		fExclusionPattern= pattern; 
		fExclusionPatternStatus.setOK();
	}
	
		
	public String getExclusionPattern() {
		return fExclusionPattern;
	}
		
	/*
	 * @see org.eclipse.jface.window.Window#configureShell(Shell)
	 */
	protected void configureShell(Shell newShell) {
		super.configureShell(newShell);
		WorkbenchHelp.setHelp(newShell, IJavaHelpContextIds.EXCLUSION_PATTERN_DIALOG);
	}
	
	// ---------- util method ------------

	private IPath chooseExclusionPattern() {
		Class[] acceptedClasses= new Class[] { IFolder.class, IFile.class };
		ISelectionStatusValidator validator= new TypedElementSelectionValidator(acceptedClasses, false);
		ViewerFilter filter= new TypedViewerFilter(acceptedClasses);

		ILabelProvider lp= new WorkbenchLabelProvider();
		ITreeContentProvider cp= new WorkbenchContentProvider();
		
		IPath initialPath= new Path(fExclusionPatternDialog.getText());
		IResource initialElement= null;
		IContainer curr= fCurrSourceFolder;
		int nSegments= initialPath.segmentCount();
		for (int i= 0; i < nSegments; i++) {
			IResource elem= curr.findMember(initialPath.segment(i));
			if (elem != null) {
				initialElement= elem;
			}
			if (elem instanceof IContainer) {
				curr= (IContainer) elem;
			} else {
				break;
			}
		}			
		String title, message;
		if (fIsExclusion) {
			title= NewWizardMessages.getString("ExclusionInclusionEntryDialog.ChooseExclusionPattern.title"); //$NON-NLS-1$
			message= NewWizardMessages.getString("ExclusionInclusionEntryDialog.ChooseExclusionPattern.description"); //$NON-NLS-1$
		} else {
			title= NewWizardMessages.getString("ExclusionInclusionEntryDialog.ChooseInclusionPattern.title"); //$NON-NLS-1$
			message= NewWizardMessages.getString("ExclusionInclusionEntryDialog.ChooseInclusionPattern.description"); //$NON-NLS-1$
		}
		
		ElementTreeSelectionDialog dialog= new ElementTreeSelectionDialog(getShell(), lp, cp);
		dialog.setTitle(title);
		dialog.setValidator(validator);
		dialog.setMessage(message);
		dialog.addFilter(filter);
		dialog.setInput(fCurrSourceFolder);
		dialog.setInitialSelection(initialElement);
		dialog.setSorter(new ResourceSorter(ResourceSorter.NAME));

		if (dialog.open() == Window.OK) {
			IResource res= (IResource) dialog.getFirstResult();
			IPath path= res.getFullPath().removeFirstSegments(fCurrSourceFolder.getFullPath().segmentCount()).makeRelative();
			if (res instanceof IContainer) {
				return path.addTrailingSeparator();
			} else {
				return path;
			}			
		}
		return null;
	}
	


}
